/*
Copyright (c) 2006-2011, Charles Jordan <skip@res.otaru-uc.ac.jp>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/
/* parse.c
 * Skip Jordan
 * Contains basic parsing stuff.
 *
 * chj	10/19/06	Created.
 * chj	11/13/06	Support for quantifiers and some commands.
 * chj	10/07/22	free_command() to fix parser memory leaks
 */

#include <stdlib.h>
#include <stdio.h>
#include "parse.h"
#include <string.h>
#include "protos.h"

struct node *parsetree;
struct node *cmdtree;

struct node *node(int l, struct node *left, struct node *right)
{
	struct node *n=malloc(sizeof(struct node));
	if (!n)
		return 0;
	n->l = left;
	n->r = right;
	n->label = l;
	n->data = 0;
	return n;
}

struct node *inode(int l, int value)
{
	struct node *n=malloc(sizeof(struct node));
	if (!n)
		return 0;
	n->label = l;
	n->data = malloc(sizeof(int));
	*((int* )(n->data)) = value; /* backwards compat, TODO:remove */
	n->ndata = value;
	n->ival = NULL;
	n->l = n->r = 0;
	return n;
}

struct node *snode(int l, char *string)
{
	struct node *n=malloc(sizeof(struct node));
	if (!n)
		return 0;
	n->label = l;
	n->data = dupstr(string);
	n->l = n->r = 0;
	n->ival = NULL;
	return n;
}

struct node *qnode(int l, struct node *vl, struct node *restr, struct node *form)
{
	struct node *n;
	struct node *vn;
        if (!(n= malloc(sizeof(struct node))))
                return 0;
        if (!(vn= malloc(sizeof(struct node))))
        {
                free(n);
                return 0;
        }

	n->label = l;
	n->l = vn;
	n->r = form;
	n->data = 0;

	vn->label=VLR;
	vn->l = vl;
	vn->r = restr;
	vn->data = 0;
	return n;
}

struct node *vlnode(int l, char *var, struct node *next)
{
	struct node *n=malloc(sizeof(struct node));
	
	if (!n)
		return 0;
	n->label = l;
	n->l = 0;
	n->r = next; 
	n->data = dupstr(var);

	return n;
}

struct node *arg_node(int l, struct node *args)
{
	struct node *n=malloc(sizeof(struct node));

	if (!n)
		return 0;
	n->label = l;
	n->l = args;
	n->r = 0;
	n->data = 0;
	return n;
}

struct node *fournode(int lab, struct node *ll, struct node *lr, struct node *rl, struct node *rr)
{
	struct node *n;
	struct node *l;
	struct node *r;

	if (!(n=malloc(sizeof(struct node))))
		return 0;

	if (!(l=malloc(sizeof(struct node))))
	{
		free(n);
		return 0;
	}
	if (!(r=malloc(sizeof(struct node))))
	{
		free(n); free(l);
		return 0;
	}

	n->label=lab;
	n->l=l;
	n->r=r;
	n->data = 0;

	l->label = FNODEIL;
	l->l = ll;
	l->r = lr;
	l->data = 0;

	r->label = FNODEIR;
	r->l = rl;
	r->r = rr;
	r->data = 0;

	return n;
}

int yyerror(char *s)
{
	printf("%s\n",s);
	return 1;
}

/* return a string of the term */
char *make_aterm_string(struct node *term)
{
	char *ret, *left, *right;
	char op;
	int len;

	switch (term->label)
	{
		case CONSTANT:
		case VAR:
			ret = dupstr(term->data);
			return ret;
		case NUMBER:
			len = snprintf(NULL, 0, "%d", term->ndata);
			ret = malloc(sizeof(char)*(len+1));
			sprintf(ret, "%d", term->ndata);
			return ret;

		case MULT:
		case PLUS:
		case MINUS:
			if (term->label==MULT)
				op='*';
			else if (term->label==PLUS)
				op='+';
			else
				op='-';
			left = make_aterm_string(term->l);
			right = make_aterm_string(term->r);

			len = snprintf(NULL, 0, "(%s)%c(%s)", left, op, right);
			ret = malloc(sizeof(char)*(len+1));
			sprintf(ret, "(%s)%c(%s)", left, op, right);
			free(left);
			free(right);
			return ret;
		default:
			printf("p01: unknown label (%d) make_aterm_string\n",
			       term->label);
			return NULL;
	}
	return NULL; /* unreachable */
}

/* free all memory included here */
void free_term_node(struct node *term)
{
	switch (term->label)
	{
		case CONSTANT:
		case VAR:
		case NUMBER:
			free(term->data);
			free(term);
			return;
		case MULT:
		case PLUS:
		case MINUS:
			free_term_node(term->l);
			free_term_node(term->r);
			free(term);
			return;
		default:
			return;
	}
}

/* frees the cmd.
 * if flag==0, LEAVES part (formulas, structures, etc) since they've
 * been assigned, etc
 * if flag==1, frees everything
 * partial implementation so far
 */
void free_command(struct node *cmd, int flag)
{

	if (cmd == NULL)
		return;

	switch (cmd->label)
	{
		case 0: /* historic: parser makes a dummy node... */
			free_command(cmd->l, flag);
			free(cmd);
			return;

		case EXPREDALL:
		case EXCONS:
			flag=1;
		case ASSIGN:
		case SAVE:
		case ABQUERY:
		case APPLY:
		case CVCONSARG:
			free_command(cmd->l, flag);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case EXPRED:
			free_command(cmd->l, 1);
			free_command(cmd->r->r, 1);
			free(cmd->r->data);
			free(cmd->r);
			free(cmd);
			return;

		case QCIR:
		case CQCIR:
		case QDIMACS:
		case PQCIR:
			free_command(cmd->l->l, flag);
			free_command(cmd->l->r, flag);
			free_command(cmd->r->l, flag);
			free(cmd->l);
			free(cmd->r);
			free(cmd);
			return ;

		case REDFIND: /* complicated because of options implementation*/
			free_command(cmd->l->l, flag);
			free_command(cmd->l->r, flag);
			free(cmd->l);
			if (cmd->r->l)
				free_command(cmd->r->l, flag);
			if (cmd->r->r && cmd->r->r->r->label==RF_RANGE)
			{
				free_command(cmd->r->r->r->l, flag);
				free_command(cmd->r->r->r->r, flag);
				free(cmd->r->r->r);
				free_command(cmd->r->r->l, flag);
				free(cmd->r->r);
			}
			else if (cmd->r->r)
			{
				free_command(cmd->r->r->l, flag);
				free_command(cmd->r->r->r, flag);
				free(cmd->r->r);
			}
			free(cmd->r);
			free(cmd);
			return;

		case FD:
			free_command(cmd->l->l, flag);
			free_command(cmd->l->r, flag);
			free(cmd->l);
			if (cmd->r->l)
			{
				free_command(cmd->r->l->l, flag);
				free_command(cmd->r->l->l, flag);
				free(cmd->r->l);
			}
			if (cmd->r->r)
			{
				free_command(cmd->r->r->l, flag);
				free_command(cmd->r->r->r, flag);
				free(cmd->r->r);
			}
			free(cmd->r);
			free(cmd);
			return;

		case MACE:
			free_command(cmd->l->l, 1);
			free_command(cmd->l->r, 1);
			if (cmd->r!=NULL)
				free_command(cmd->r, 1);
			free(cmd->l);
			free(cmd);
			return;

		case BQUERY:
			free_command(cmd->l, flag);
			if (flag)
				free_command(cmd->r, flag);
			free(cmd);
			return;

		case REDUC:
			free_command(cmd->l->l->l, 1);
			free_command(cmd->l->l->r, 1);
			free(cmd->l->l);
			free_command(cmd->l->r, flag); /* crreldefs */
			free(cmd->l);
			free_command(cmd->r->l, 1);
			if (cmd->r->r)
				free_command(cmd->r->r, flag); /* formula */
			free(cmd->r);
			free(cmd);
			return;

		case CRRELDEF:
			free_command(cmd->l->l, 1);
			free_command(cmd->l->r, 1);
			free(cmd->l);
			free_command(cmd->r->r, flag);
			if (flag)
				free_command(cmd->r->l, flag);
			free(cmd->r);
			free(cmd);
			return;

		case VOCAB:
		case STRUC:
			free_command(cmd->l, flag);
			free(cmd);
			return;

		case CVRELARG:
		case CSARGS:
			free_command(cmd->l->l, 1);
			free_command(cmd->l->r, 1);
			free(cmd->l);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case CSRELDEF:
			free_command(cmd->l->l, 1);
			free_command(cmd->l->r, 1);
			free(cmd->l);
			free_command(cmd->r->r, flag);
			if (flag==1)
				free_command(cmd->r->l, flag);
			free(cmd->r);
			free(cmd);
			return;

		case CSCONSDEF:
			free_command(cmd->l->l, 1);
			free_command(cmd->l->r, 1);
			free(cmd->l);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case CRCONSDEF:
			free_command(cmd->l->l, 1);
			if (flag==1)
				free_command(cmd->l->r, flag);
			free(cmd->l);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case RELARGS:
			free_command(cmd->l, flag);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case TRUE:
		case FALSE:
			if (flag==1)
				free(cmd);
			return;

		case ID:
		case FILENAME:
		case LOAD:
		case CONSTANT:
		case NUMBER:
		case VAR:
			free(cmd->data);
			free(cmd);
			return;
		case PRED:
			free(cmd->data);
			free_command(cmd->r, flag);
			free(cmd);
			return;
		case SOE:
		case FORALL:
		case EXISTS:
		case TC:
		case TCARGS:
			if (cmd->label==TCARGS)
				flag = 1;
			if (flag == 0)
				return;
			free_command(cmd->l->l, flag);
			free_command(cmd->l->r, flag);
			free(cmd->l);
			free_command(cmd->r, flag);
			free(cmd);
			return;

		case VARLIST:
			free(cmd->data);
			free_command(cmd->r, 1);
			free(cmd);
			return;

		case AND:
		case OR:
		case IFF:
		case LT:
		case LTE:
		case IMPLIES:
		case XOR:
		case EQUALS:
		case NEQUALS:
		case PLUS:
		case MINUS:
			if (flag==0)
				return;
			free_command(cmd->l, flag);
			free_command(cmd->r, flag);
			free(cmd);
			return;
		case NOT:
			if (flag==0)
				return;
			free_command(cmd->l, flag);
			free(cmd);
			return;

		default:
#ifdef DEDEBUG
			printf("free_command(): unhandled label %d\n", cmd->label);
#endif
			return;
	}
}
