/*
Copyright (c) 2013, Charles Jordan <skip@res.otaru-uc.ac.jp>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/
/* externalsat.c
 * Skip Jordan
 *
 * Based on the C bindings for MiniSat2 (minisat.cc)
 *
 * Mimic MiniSat incremental API, but call an external SAT solver
 * instead (non-incrementally).
 *
 * This is unfortunately not currently portable (POSIX only).  Not
 * possible in strict C89, so we use popen.
 *
 * We assume that solver.sh is a script running a solver in the path.
 * It should take two arguments, solver.sh file.dimacs output should
 * run the solver on file.dimacs and produce MiniSat-style output
 * in output (i.e., Lukasz's solver.sh script).  This allows to easily
 * change the external solver without recompiling.
 *
 * This allows us to use GPL SAT solvers, since many interpret the GPL to
 * require us to be GPL if linking dynamically.  We execute the solver as
 * a separate process, giving us the freedom to license as we desire.
 *
 * chj	07/18/13	created to use treengeling/etc.
 */

/* Based on minisat.cc, original copyright:
Copyright (c) 2008-2010, Niklas Sorensson
              2008, Koen Claessen

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
associated documentation files (the "Software"), to deal in the Software without restriction,
including without limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or
substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT
OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdlib.h>
#include <stdio.h>
#include "minisat.h"
#include <string.h>
#include <assert.h>
#include "protos.h"

struct cur_clause {
	int lit;
	struct cur_clause *next;
};

struct clause {
	int *lits; /* lits[0]\lor lits[1] \lor ... \lor lits[length-1] */
	int length; /* number of lits */
	struct clause *next;
};

struct minisat_solver_t {
	struct cur_clause *cur;
	struct clause *clauses;
	struct clause *last_clause;
	int *eval;
	int result;
	int curVar;
	int num_clauses;
};

minisat_solver *extsat_parseoutput(minisat_solver *sv, char *filename);
char *extsat_makedimacs(minisat_solver *sv, int len, minisat_Lit *assumps);
minisat_solver *extsat_solve(minisat_solver *sv, char *instance);

extern const minisat_lbool minisat_l_True = 1;
extern const minisat_lbool minisat_l_False = 0;
extern const minisat_lbool minisat_l_Undef = -1;

minisat_solver *minisat_new(void)
{
	struct minisat_solver_t *sv = malloc(sizeof(struct minisat_solver_t));
	sv->cur = NULL;
	sv->clauses=NULL;
	sv->eval=NULL;
	sv->result=-1;
	sv->curVar=0;
	sv->num_clauses=0;
	return sv;
}

void minisat_delete(minisat_solver *sv)
{
	struct cur_clause *ccl, *tm;
	struct clause *cl, *tm2;
	
	for (ccl=sv->cur; ccl; ccl=tm)
	{
		tm=ccl->next;
		free(ccl);
	}

	for (cl=sv->clauses; cl; cl=tm2)
	{
		tm2=cl->next;
		if (cl->lits)
			free(cl->lits);
		free(cl);
	}

	if (sv->eval)
		free(sv->eval);

	free(sv);
	return;
}

minisat_Var minisat_newVar(minisat_solver *sv) { return ++(sv->curVar); }
minisat_Lit minisat_newLit(minisat_solver *sv) { return ++(sv->curVar); }
minisat_Lit minisat_mkLit(minisat_Var x) { return x; }
minisat_Lit minisat_negate(minisat_Lit p) { return (-1)*p; }
minisat_Var minisat_var(minisat_Lit p) { return p>=0?p:(-1)*p; }

void minisat_addClause_begin(minisat_solver *sv)
{
	assert(sv->cur == NULL);
	return;
}

void minisat_addClause_addLit(minisat_solver *sv, minisat_Lit p)
{
	struct cur_clause *cur=malloc(sizeof(struct cur_clause));
	assert(p<=sv->curVar && (-1)*p<=sv->curVar);
	cur->lit=p;
	cur->next = sv->cur;
	sv->cur=cur;
	return;
}

minisat_bool minisat_addClause_commit(minisat_solver *sv)
{
	int length, i;
	struct cur_clause *cur, *tm;
	struct clause *clause;
	int *lits;

	for (length=0,cur=sv->cur; cur; cur=cur->next,length++);
	assert(length>0);

	clause = malloc(sizeof(struct clause));
	lits = clause->lits = malloc(sizeof(int)*length);
	clause->length=length;

	for (i=0,cur=sv->cur; i<length; i++,cur=cur->next)
		lits[i]=cur->lit;

#ifdef EXTSAT_LASTFIRST
	clause->next = sv->clauses;
	sv->clauses = clause;
#else /* DEFAULT */
	clause->next=NULL;
	if (sv->clauses)
	{
		sv->last_clause->next=clause;
		sv->last_clause=clause;
	}
	else
		sv->last_clause = sv->clauses = clause;
#endif

	for (cur=sv->cur; cur; cur=tm)
        {
                tm=cur->next;
                free(cur);
        }

	sv->cur = NULL;
	return 1;
}

minisat_lbool minisat_modelValue_Lit (minisat_solver *sv, minisat_Lit p)
{
	assert(p<=sv->curVar && (-1)*p<=sv->curVar);
	assert(sv->result == 1); /* satisfiable, guarantees eval[|p|] okay */
	return (p>=0?sv->eval[p]:sv->eval[(-1)*p]);
}

/* we fake incremental for now so no need to freeze anything */
void minisat_setFrozen(minisat_solver* sv, minisat_Var v, minisat_bool b) 
	{ return; }

int minisat_solve(minisat_solver *sv, int len, minisat_Lit *assumps)
{
	char *instance = dupstr(extsat_makedimacs(sv, len, assumps));
	sv = extsat_solve(sv, instance);
#ifndef EXTSAT_KEEPDIMACS
	remove(instance);
#endif
	free(instance); /* the filename :-P */

	return sv->result;
}

/* instance is the dimacs for sv->clauses (maybe with assumptions)
 * solve it with an external solver, parse the output and put the
 * result in sv->result (0 unsat, 1 sat).
 * If SAT, then put the model in sv->eval[...], where
 * sv[i] is the evaluation of i (note: 1 <= i <= sv->curVar)
 * (malloc sv->eval if SAT, set sv->eval=NULL if UNSAT)
 */
minisat_solver *extsat_solve(minisat_solver *sv, char *instance)
{
	char *output;
	char *command;
	FILE *p;
	int c;

	output = tmpnam(NULL);
	command = malloc(sizeof(char)*(10+strlen(instance)+1+strlen(output)+1));
	sprintf(command,"solver.sh %s %s",instance,output);

#ifdef REDFIND_DEBUG
	printf("\n	solver.sh %s %s\n\n",instance,output);
#endif

	p = popen(command,"r");
	if (!p)
	{
		err("es2: Unable to execute solver.sh\n");
#ifndef	EXTSAT_KEEPDIMACS
		remove(instance);
#endif
		free(command);
		return sv;
	}

	while ((c=getc(p))!=EOF);

	pclose(p);
	
	sv = extsat_parseoutput(sv, output);

#ifndef EXTSAT_KEEPDIMACS
	remove(instance);
#endif
	remove(output);
	free(command);
	return sv;
}

/* filename has MiniSat-style output hopefully.
 * read it and put it in sv->result / sv->eval
 */
minisat_solver *extsat_parseoutput(minisat_solver *sv, char *filename)
{
	FILE *out;
	int var;
	int *eval;

	int i;
	int c;
	int numvars;

	out = fopen(filename,"r");
	if (!out)
	{
		printf("es3: Unable to open output of SAT solver\n");
		sv->result = -1;
		return sv;
	}

	for (i=getc(out); i!='S' && i!='U' && i!=EOF; i=getc(out));

	if (i==EOF)
	{
		printf("es4: SAT solver output ended unexpectedly\n");
		sv->result = -1;
		return sv;
	}

	if (i=='U')
	{
		sv->result = 0;
		/* TODO : check that it's actually UNSAT and not just a spurious U :-P */
		fclose(out);
		if (sv->eval)
			free(sv->eval);
		sv->eval = NULL;
		return sv;
	}

	/* otherwise we just read the S of SAT */
	if (getc(out)!='A')
	{
		printf("es5: SAT said 'S' not \"SAT\"\n");
		sv->result = -1;
		fclose(out);
		return sv;
	}

	if (getc(out)!='T')
	{
		printf("es6: SAT said 'S' not \"SAT\"\n");
                sv->result = -1;
                fclose(out);
                return sv;
        }

	getc(out); /* whitespace */

	/* okay, now at the right spot to read model */
	numvars = sv->curVar;
	if (sv->eval)
		free(sv->eval);
	eval = sv->eval = malloc(sizeof(int)*(numvars+1));

	for (i=0; i<=numvars; i++)
		eval[i]=-1; /* to check later */
	sv->result = 1;

	while ((c=fscanf(out,"%d",&var))!=EOF)
	{
		if (c==0)
		{
			printf("es7: error parsing SAT output\n");
			fclose(out);
			return sv;
		}
		assert(var<=numvars && (-1)*var<=numvars);
		eval[(var>0)?var:(-1)*var]=(var>0?1:0);
	}

#ifdef REDFIND_DEBUG
	for (i=1; i<=numvars; i++)
	{
		if (eval[i]<0 || eval[i]>1)
			printf("es8: SAT output doesn't assign all variables?\n");
	}
#endif
	fclose(out);
	return sv;
}

/* create a temporary file with a dimacs instance corresponding to the
 * clauses in sv->clauses, with additional len-many assumps
 * fclose() and return filename
 */
char *extsat_makedimacs(minisat_solver *sv, int len, minisat_Lit *assumps)
{
	char *filename;
	FILE *file;
	int num_clause;
	struct clause *clause;
	int i, clen;

	for (num_clause=0, clause=sv->clauses; clause; num_clause++, clause=clause->next);
	/* NOTE : num_clause does not include the assumptions! */

	filename = tmpnam(NULL);
	file = fopen(filename, "w");	

	if (!file)
	{
		err("es1: Can't open temporary file %s for writing.\n", filename);
		if (filename)
			free(filename);
		return NULL;
	}

	fprintf(file, "c produced by de : externalsat.c\n");
	fprintf(file, "p cnf %d %d\n", sv->curVar, num_clause + len);

	for (clause = sv->clauses; clause; clause=clause->next)
	{
		clen = clause->length;
#ifndef EXTSAT_LASTFIRST
		for (i=clen-1; i>=0; i--)
#else
		for (i=0; i<clen; i++)
#endif
			fprintf(file, " %d",clause->lits[i]);
		fprintf(file, " 0\n");
	}

#ifndef EXTSAT_LASTFIRST
	for (i=len-1; i>=0; i--)
#else
	for (i=0; i<len; i++)
#endif
		fprintf(file, " %d 0\n",assumps[i]);

	fclose(file);
	return filename;
}
