/*
Copyright (c) 2020-2025, Charles Jordan <skip@res.otaru-uc.ac.jp>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/
/* trans.c
 * Skip Jordan
 *
 * Translating model-checking SO(TC) to QCIR
 * possibly more later?
 *
 * chj  2/23/20         created
 * chj	5/8/22		basics done, no TC yet
 * chj	5/20/22		TC1 without added variables
 * chj	8/12/22		support for general TC (not only TC1)
 * chj	8/16/22		simple use of variable dependencies for TC
 * chj	8/23/22		poly-size encoding for TC2
 * chj 11/21/22		output prenex QCIR in prenex format
 * chj	3/29/25		fixes to TC2 applied to predicates in the structure
 *
 * last output number: tr36 
 * todo: better tracking of dependencies for TC especially. currently we
 * track dependency on a quantifier block, but could do better (individual
 * variables or re-use again if the same assignments are present).
 */

#include "parse.h"
#include "protos.h"
#include "hash.h"
#include <stdio.h>
#include <string.h>
#include "deqbf.h"
#include <stdlib.h>
#include <math.h>
#include <assert.h>

struct dagnode *to_qcir_rec(struct structure *, struct node *, struct tr *,
			    struct interp *);
int check_tr_form(struct structure *, struct node *);
void make_tr_tcsovc(struct tr *, struct structure *, struct node *);
struct dagnode *to_qcir(struct structure *, struct node *, FILE *, unsigned long *, int);
void dag_cleanup(struct dagnode *);
void qcir_to_file(struct dagnode *, FILE *, unsigned long, int);
void free_renames(struct relrename *);
void free_sovc(struct tr *);
struct dagnode *old_get_tr_tcdagnode(struct structure*,struct node*,struct tr*,
				 struct interp*,struct tcvc*,unsigned long long,
                                 int *, char **, int *, char **);
void make_tcdeps_rec(struct tr *, struct structure *, struct node *,struct node *,
		     char **, int);
struct dagnode *dag_simplify(struct tr *tr, struct dagnode *);

/* hack for truth constants, to allow use of existing tools */
/* looks like fmla and qcir-conv.py don't need this: so default is off */
/* re-used for qdimacs conversion to get true/false constants if needed */
struct dagnode dagn_true, dagn_false;
struct dagnode *qcirhack(struct tr *, struct dagnode *);
struct dagnode *qcirhack_replace(struct tr *, struct dagnode *);

/* convert struc \models form to QCIR and output to given filename */
/* command_type is QCIR, CQCIR (cleansed), PQCIR (prenex cleansed), QDIMACS */
/* overloaded a bit, but they're all produced in a similar way from QCIR */
int qcir_driver(struct structure *struc, struct bquery *q, const char *filename,
		int command_type)
{
	struct node *form = q->form;
	struct dagnode *dag;
	FILE *f;
	char *fn; /* parser has extra quotation marks in fn */
	unsigned long n_v = 0; /* for cleansed QCIR output */

	fn=dupstr(filename+1);
	fn[strlen(fn)-1]='\0';

	f = fopen(fn, "w");
	if (f==NULL)
	{
		printf("tr5: Unable to open file %s for writing\n", fn);
		free(fn);
		return 0;
	}

	tr_debug4("trd: translating %s \\models %s to QBF, file %s\n",
		  struc->name, q->name, fn);
	/* to_qcir does file output only if command_type is QDIMACS */
	dag = to_qcir(struc, form, f, &n_v, command_type);

	if (command_type!=QDIMACS && dag!=NULL)
	{
		tr_debug2("trd: finished dag conversion, next is output to %s\n", fn);
		qcir_to_file(dag, f, n_v, command_type);
	}

	fclose(f);

	tr_debug1("trd: finished, cleaning up and returning\n");
	if (dag!=NULL)
		dag_cleanup(dag);
	free(fn);
	free(dagn_true.vars[0]);
	free(dagn_true.vars);
	free(dagn_true.neg);
	free(dagn_false.vars[0]);
	free(dagn_false.vars);
	free(dagn_false.neg);
	return 0;
}


struct dagnode *qcirhack(struct tr *tr, struct dagnode *dag)
{
	struct dagnode *ret = calloc(1, sizeof(struct dagnode));
	struct dagnode *body[3];
	int neg[3]={0};

	body[0] = &dagn_true;
	body[1] = &dagn_false;
	body[2] = dag;
	neg[1] = 1;

	ret->type = TR_EXISTS;
	ret->num = tr->hash->next_id++;
	ret->child = malloc(sizeof(struct dagnode*));
	ret->child[0] = tr_get_dagnode_v(tr, TR_AND, body, neg, 3);
	ret->n_child = 1;
	ret->n_vars = 2;
	ret->vars = malloc(sizeof(char *)*2);
	ret->vars[0] = dupstr(dagn_true.vars[0]);
	ret->vars[1] = dupstr(dagn_false.vars[0]);

	return ret;
}

void make_dagn_tf(struct tr *tr)
{
	struct dagnode *d, *arr[2]={&dagn_true,&dagn_false};
	int i, len;

	for (i=0; i<2; i++)
	{
		d=arr[i];
		d->type = TR_VAR;
		d->num = tr->hash->next_id++;
		d->child = NULL;
		d->n_child = 0;
		d->vars = malloc(sizeof(char *));
		len = snprintf(NULL, 0, "%d", d->num)+1;
		d->vars[0] = malloc(sizeof(char)*len);
		sprintf(d->vars[0], "%d", d->num);
		d->n_vars = 1;
		d->neg = malloc(sizeof(int));
		d->neg[0] = 0;
		d->printed = 2;
	}
}

void free_tcvc(struct tr *tr)
{
	int i;
	for (i=0; i<tr->n_tcvc; i++)
	{
		free(tr->tcvc[i].sorts);
		free(tr->tcvc[i].values1);
		free(tr->tcvc[i].values2);
		free(tr->tcvc[i].renames1);
		free(tr->tcvc[i].renames2);
	}

	free(tr->tcvc);
}

void free_tcdeps(struct tr *tr)
{
	int i;
	for (i=0; i<tr->n_tcdeps; i++)
		free(tr->tcdeps[i].tcvc);
	free(tr->tcdeps);
}

/* produce the (parsed) formula
 * "_Mc(s) & \forall _y : (_Mc(_y)->_y=s)"
 * note that this _Mc and _y are not available through the parser -
 * so no need to worry about conflicts with user symbols.
 * we hack these symbols into the parser's output..
 */
struct node *tr_equiv_mso_form(int c, const char *s)
{
	struct node *node, *pred1, *pred2, *quant, *vn;
	char *form, *tmp;
	void *bufstate; /* YY_BUFFER_STATE */
	int len;

	len = snprintf(NULL, 0, "internal is new bquery{graph, Mm%d(%s)&\\forall yy:(Mm%d(yy)->yy=%s)}.\n",c,s,c,s);
	form = malloc(sizeof(char)*(len+1));
	sprintf(form, "internal is new bquery{graph, Mm%d(%s)&\\forall yy:(Mm%d(yy)->yy=%s)}.\n",c,s,c,s);
	bufstate = yy_scan_string(form);
	yyparse();
	node = cmdtree->l->r->r;
	yy_delete_buffer(bufstate);
	free(form);

	pred1 = node->l; /* Mmc(yy), change to _Mc(s) */
	pred2 = node->r->r->l; /* Mmc(yy), change to _Mc(_y) */
	quant = node->r; /* change variable yy to _y */
	vn = node->r->r->r->l; /* change variable yy to _y */
	
	assert(pred1->label==PRED && pred2->label==PRED &&
	       quant->label==FORALL && vn->label==VAR);

	tmp=pred1->data;
	tmp[0] = '_';
	tmp[1] = 'M';

	tmp=pred2->data;
	tmp[0] = '_';
	tmp[1] = 'M';
	tmp=pred2->r->l->data;
	tmp[0] = '_';

	tmp=quant->l->l->data;
	tmp[0] = '_';

	tmp=vn->data;
	tmp[0] = '_';

	free_command(cmdtree->l, 0);
	free(cmdtree);
	return node;
}

/* form is a TC, already handled body.
 * if not TC1/TC2, convert to TC2 - handling relargs tcargs and tcform.
 */
struct node *trans_tc_form_direct(struct node *form)
{
	struct node *tcargs, *relargs, *tmp, *tcform;
	struct node *newnode, *varlist, *quant, *restr, *ret;
	char **vnames; /* we replace these "x1,y1, ..., xr,yr" */
	char **prednames; /* using these predicate symbols */
	int len;
	int i, c = 0; /* used to get non-conflicting local variable names */
	int twor=0; /* r times two, length of vnames */
#ifndef TR_CONVERT_TC1
	if (!check_tc_mixed(form))
		return form;
#else
	if (!check_tc_tc1(form) && !check_tc_mixed(form))
		return form;
#endif
	/* otherwise need to convert */

	tcargs = form->l->l;
	relargs = form->r;
	tcform = form->l->r;

	for (tmp=tcargs; tmp; tmp=tmp->r)
	{
		if (tmp->l->l->label != PREDVAR)
			twor++;
		if (tmp->l->r->label != PREDVAR)
			twor++;
	}

	assert(twor>0); /* otherwise check_tc_mixed check_tc_tc1 would abort */

	vnames = malloc(sizeof(char *) * twor);
	prednames = malloc(sizeof(char *) * twor);

	/* first, replace the first-order TC arguments using monadic symbols */
	for (tmp=tcargs; tmp; tmp=tmp->r)
	{
		if (tmp->l->l->label != PREDVAR)
		{
			len = snprintf(NULL, 0, "_M%d", c);
			prednames[c] = malloc(sizeof(char)*(len+1));
			sprintf(prednames[c], "_M%d", c);
			newnode = node(PREDVAR, snode(PRED, prednames[c]),
				       inode(NUMBER, 1));
			vnames[c] = tmp->l->l->data;

			free(tmp->l->l);
			tmp->l->l = newnode;
			c++;
		}
		if (tmp->l->r->label != PREDVAR)
		{
			len = snprintf(NULL, 0, "_M%d", c);
			prednames[c] = malloc(sizeof(char)*(len+1));
			sprintf(prednames[c], "_M%d", c);
			newnode = node(PREDVAR, snode(PRED, prednames[c]),
				       inode(NUMBER, 1));
			vnames[c] = tmp->l->r->data;

			free(tmp->l->r);
			tmp->l->r = newnode;
			c++;
		}
	}

	/* next, modify the tcform (\phi) to re-introduce first-order symbols
	 * using the monadic symbols
	 */
	newnode = NULL;
	varlist = NULL;
	tmp = NULL;

	for (i=0; i<twor; i++) /* make FO variable list */
	{
		newnode = vlnode(VARLIST, vnames[i], NULL);
		if (varlist == NULL)
			varlist = tmp = newnode;
		else
		{
			tmp->r = newnode;
			tmp = newnode;
		}
	}
	restr = NULL;
	tmp = NULL;
	newnode = NULL;
	for (i=0; i<twor; i++) /* make FO restriction */
	{
		newnode = tr_equiv_mso_form(i, vnames[i]);
		if (restr == NULL)
		{
			if (twor==1) /* can't happen but handle it anyway */
			{
				restr = newnode;
				break;
			}
			restr = tmp = node(AND, newnode, NULL);
		}
		else
		{
			if (i+1==twor) /* last one, don't make another AND */
				tmp->r = newnode;
			else /* have another to make, make an AND */
			{
				tmp->r = node(AND, newnode, NULL);
				tmp = tmp->r;
			}
		}
		
	}

	/* create \phi' using \phi */
	quant = qnode(EXISTS, varlist, restr, tcform);
	form->l->r = quant;
	free(vnames);
	for (i=0; i<c; i++)
		free(prednames[i]);
	free(prednames);

	/* replace FO relargs with new monadic predicates
	 * and introduce leading SO existential quantifiers
	 */
	twor = 0;
	for (tmp=relargs; tmp; tmp=tmp->r)
	{
		if (tmp->label == TCRELARGS)
			continue;
		twor++;
	}
	vnames = malloc(sizeof(char *)*twor);
	prednames = malloc(sizeof(char *)*twor);

	tmp=relargs;
	for (i=0; i<twor; tmp=tmp->r)
	{
		if (tmp->label == TCRELARGS)
			continue;
		vnames[i] = make_aterm_string(tmp->l);
		len = snprintf(NULL, 0, "_M%d", i+c);
		prednames[i] = malloc(sizeof(char)*(len+1));
		sprintf(prednames[i], "_M%d", i+c);
		i++;
	}

	quant = NULL;
	tmp = NULL;
	newnode = NULL;
	/* make leading SO block */
	for (i=0; i<twor; i++)
	{
		newnode = qnode(SOE, qnode(SOVARLIST, snode(PRED,prednames[i]), 
					   inode(NUMBER, 1), 0),
				tr_equiv_mso_form(i+c, vnames[i]),
				NULL);
		if (quant == NULL)
			quant = tmp = newnode;
		else
		{
			tmp->r = newnode;
			tmp = newnode;
		}
	}
	tmp->r = form; /* original/converted TC goes here */
	ret = quant;
	
	i=0;
	/* convert original relargs to use prednames */
	for (tmp=relargs; tmp; tmp=tmp->r)
	{
		if (tmp->label == TCRELARGS)
			continue;
		/* convert tmp->l from it's current term to use prednames[i] */
		/* free the memory no longer used in tmp->l */
		free_term_node(tmp->l);
		tmp->l = snode(PRED, prednames[i++]);
		tmp->label = TCRELARGS;
	}
	for (i=0; i<twor; i++)
	{
		free(vnames[i]);
		free(prednames[i]);
	}
	free(vnames);
	free(prednames);

	return ret;
}

/* modify the parsed formula form, to translate any mixed TC to
 * strict TC2, by replacing first-order symbols with monadic
 * predicates.  this must be called BEFORE setting up fast
 * interpretations, sovc, etc.
 */
struct node *trans_tc_form(struct tr *tr, struct node *form)
{
	if (form == NULL)
		return NULL;

	switch (form->label)
	{
		case TRUE:
		case FALSE:
		case NUMBER:
		case LT:
		case LTE:
		case NEQUALS:
		case EQUALS:
		case MULT:
		case PLUS:
		case MINUS:
		case PRED:
			return form;
		case NOT:
			form->l = trans_tc_form(tr, form->l);
			return form;
		case AND:
		case OR:
		case IMPLIES:
		case IFF:
		case XOR:
			form->l = trans_tc_form(tr, form->l);
			form->r = trans_tc_form(tr, form->r);
			return form;
		case EXISTS:
		case FORALL:
		case SOE:
			form->r = trans_tc_form(tr, form->r);
			if (form->l->r)
				form->l->r = trans_tc_form(tr, form->l->r);
			return form;
		case IFP:
			printf("tr34: IFP unsupported in qcir()\n");
			tr->abort = 1;
			return form;
		case TC:
			/* tcform: trans_tc_form can handle nested TC */
			form->l->r = trans_tc_form(tr, form->l->r);
			form = trans_tc_form_direct(form);
			return form;
		default:
			printf("tr35: unknown node type %d in trans_tc_form\n",
			       form->label);
			return form;
	}
	return form; /* unreachable */
}


/* convert "struc\models form" to a qcir dag */
/* command_type is QCIR (normal), CQCIR (cleanse), PQCIR (prenex and cleanse),
 * QDIMACS (prenex, cleanse, to CNF, and output to file f)
 * for QCIR output is separate...
 */
struct dagnode *to_qcir(struct structure *struc, struct node *form,
			FILE *f, unsigned long *n_v, int command_type)
{
	struct tr *tr;
	struct env *hash;
	struct dagnode *ret;
	struct interp *interp;

	if (!check_tr_form(struc, form))
		return NULL;

	tr = malloc(sizeof(struct tr));
	tr->keys = NULL;
	tr->sovc = NULL;
	tr->n_sovc = 0;
	tr->tcvc = NULL;
	tr->n_tcvc = 0;
	tr->renames = NULL;
	tr->tcdeps = NULL;
	tr->n_tcdeps = 0;
	tr->unused = NULL;
	hash = malloc(sizeof(struct env));
	interp  = new_interp(struc);
	tr->abort = 0;
	tr->warn = 0;

	tr->hash = hash;
	hash->id_hash = hash_create(TR_MAXIDS, (hash_comp_t)strcmp, 0);
	hash->next_id = 1;
	tr->hits = 0;

	tr->qcirhack = 0;
	if (command_type == QDIMACS) /* makes life easier for qdimacs */
		tr->qcirhack = 1;
#ifdef QCIRHACK
	tr->qcirhack = 1;
#endif

	form = trans_tc_form(tr, form);

	eval_init_form(form, interp, struc); /* from eval.c */
	make_tr_tcsovc(tr, struc, form);
	make_dagn_tf(tr);

	/* call simple recursive making version, using hashtable on
	 * gates to avoid silly duplication
	 */
	ret = to_qcir_rec(struc, form, tr, interp);

	if (tr->qcirhack)
	{
		tr_debug1("trd: hacking in true/false definitions\n");
		ret = qcirhack(tr, ret);
	}

	tr_debug1("trd: finished circuit construction, simplifying\n");
	ret = dag_simplify(tr, ret);

	if (command_type != QCIR)
	{
		tr_debug1("trd: cleansing\n");
		ret = qcir_cleanse(tr, ret, n_v);
	}
	if (command_type == PQCIR || command_type == QDIMACS)
	{
		tr_debug1("trd: prenexing\n");
		ret = qcir_prenex(tr, ret, n_v);
	}
	if (command_type == QDIMACS)
	{
		qcir_to_qdimacs(tr, ret, f, "de conversion from SO(TC)");
	}

	hash_free_nodes(hash->id_hash);
	hash_destroy(hash->id_hash);
	free_interp(interp);
	free_list(tr->keys);
	free(hash);
	free_sovc(tr);
	free_tcvc(tr);
	free_tcdeps(tr);
	free_renames(tr->renames);
	printf("tr0: finished conversion, %lld cache hits\n", tr->hits);
	free(tr);

	return ret;
}

/* TODO: check arity of things, check bound variables?
 * in any case, for now: require no LFP, check TC is okay
 * forbid nested TC (flag notes whether we're inside a TC)
 */
int check_tr_form_rec(struct structure *struc, struct node *form,
		      struct interp *interp, int flag)
{
	struct relation *rel=NULL; /* initialization is just warning removal */
	int left=1, right=1;
	if (struc == NULL || form == NULL)
		return 0; /* unreachable, but warning removal */
	switch (form->label)
	{
		case TRUE:
		case FALSE:
		case NUMBER:
		case LT:
		case LTE:
		case NEQUALS:
		case EQUALS:
		case PLUS:
		case MINUS:
		case MULT:
		case CONSTANT: /* TODO: check in vocabulary */
		case VAR:      /* TODO: check bound */
		case PRED:     /* TODO: check in vocabulary or bound */
			return 1;
		case NOT:
			return check_tr_form_rec(struc, form->l,interp,flag);
		case AND:
		case OR:
		case IMPLIES:
		case IFF:
		case XOR:
			return check_tr_form_rec(struc,form->l,interp,flag)&&
			       check_tr_form_rec(struc, form->r,interp,flag);
		case SOE:
			/* put SOE predicate in interpretation, to check
			 * matching arity in TC uses
			 */
			rel = malloc(sizeof(struct relation));
			rel->name = form->l->l->l->l->data;
			rel->arity=*(int *)(form->l->l->l->r->data);
			rel->next = interp->rel_symbols;
			rel->cache = NULL;
			rel->parse_cache = 0;
			interp->rel_symbols = rel;
			/* falls through to below */
		case FORALL:
		case EXISTS:
			if (form->l->r != NULL) /* restriction */
				left = check_tr_form_rec(struc, form->l->r,
							 interp, flag);
			right = check_tr_form_rec(struc,form->r,interp,flag);
			if (form->label == SOE) /* remove from interp */
			{
				interp->rel_symbols = rel->next;
				free(rel);
			}
			return left && right;
		case TC:
			if (check_tc_sorts(form, interp, struc)==0)
			{
				printf("tr24: bad TC usage\n");
				return 0;
			}
			if (flag)
			{
				printf("tr25: nested TC not currently supported for formula translation\n");
				return 0;
			}
			return check_tr_form_rec(struc, form->l->r,interp,1);
		case IFP:
			printf("tr22: fixed points not supported for qcir translation\n");
			return 0;

		default:
			printf("tr23: unknown logical node (%d)\n", form->label);
			return 0;
	}

	return 0; /* unreachable */
}

int check_tr_form(struct structure *struc, struct node *form)
{
	struct interp *interp = new_interp(struc);
	int ret = check_tr_form_rec(struc, form, interp, 0);
	free_interp(interp);
	return ret;
}

/* if there is a relrename for oldname, return it
 * otherwise, make a new one, add to tr, and return it.
 */
struct relrename *get_tc_relname(struct tr *tr, char *oldname)
{
	struct relrename *rr;
	for (rr=tr->renames; rr; rr=rr->next)
		if (!strcmp(rr->oldname, oldname))
			return rr;
	rr = malloc(sizeof(struct relrename));
	rr->oldname = oldname;
	rr->newname = NULL;
	rr->next = tr->renames;
	tr->renames = rr;
	return rr;
}

/* form is a TC formula; add it to tcvc as needed */
void make_tr_tcvc(struct tr *tr, struct node *form)
{
	struct tcvc *tmp;
	struct node *tnode, *tcargs=form->l->l;
	int **values1, **values2;
	int *sorts;
	struct relrename **renames1, **renames2;
	int i, n_tcvc=tr->n_tcvc, tup_arity;

	for (i=0; i<n_tcvc; i++)
		if (tr->tcvc[i].tcform == form)
			return; /* shouldn't happen, but it's already there */

	tmp = realloc(tr->tcvc, sizeof(struct tcvc)*(n_tcvc+1));
	if (tmp == NULL)
	{
		tr->abort = 1;
		printf("tr6: unable to allocate memory\n");
		return;
	}
	tr->tcvc = tmp;
	tmp = tr->tcvc+n_tcvc;
	tmp->tcform = form;
	tmp->num = n_tcvc;
	tmp->cur = 0;
	tmp->counter = 1;
	tr->n_tcvc++;

	/* set up convenience things */
	sorts = get_sorts(form);
	tnode=tcargs;
	for (tup_arity=0,tnode=tcargs; tnode; tup_arity++)
		tnode=tnode->r;

	tmp->tup_arity=tup_arity;
	tmp->sorts=sorts;
	values1 = calloc(tup_arity, sizeof(int *));
	values2 = calloc(tup_arity, sizeof(int *));
	renames1 = calloc(tup_arity, sizeof(struct relrename *));
	renames2 = calloc(tup_arity, sizeof(struct relrename *));

	tmp->values1 = values1;
	tmp->values2 = values2;
	tmp->renames1 = renames1;
	tmp->renames2 = renames2;

	tnode=tcargs;
	for (i=0; i<tup_arity; tnode=tnode->r)
	{
		if (sorts[i]==-1)
			values1[i] = tnode->l->l->ival;
		else
			renames1[i] =get_tc_relname(tr,tnode->l->l->l->data);
		if (++i<tup_arity)
		{
			if (sorts[i]==-1)
				values1[i] = tnode->l->r->ival;
			else
				renames1[i] = get_tc_relname(tr,
						       tnode->l->r->l->data);
			i++;
		}
		else
			break;
	}
	if (tup_arity&1) /* odd arity means we split one */
	{
		if (sorts[0]==-1)
			values2[0] = tnode->l->r->ival;
		else
			renames2[0] =get_tc_relname(tr,tnode->l->r->l->data);
		i=1;
		tnode=tnode->r;
	}
	else
		i=0;

	for (; ; tnode=tnode->r)
	{
		if (!tnode)
			break;
		if (sorts[i]==-1)
			values2[i] = tnode->l->l->ival;
		else
			renames2[i] =get_tc_relname(tr,tnode->l->l->l->data);
		if (++i<tup_arity)
		{
			if (sorts[i]==-1)
				values2[i] = tnode->l->r->ival;
			else
				renames2[i] = get_tc_relname(tr,
						       tnode->l->r->l->data);
			i++;
		}
		else
			break;
	}
	return;
}

/* we're making tcdeps for formula orig.
 * it binds the variables in varnames.
 * form is a FO or SO quantifier - if it binds any names in varnames,
 * we want to make a copy of varnames minus the rebound ones and recurse
 * otherwise, just recurse.
 * this is because, if the TC depends on any such variables, that dependency
 * is on form, not on orig, so we don't want to add a dependency on orig.
 */
/* we also need to check the restriction if it exists */
/* if form is a second order quantifier, it only has one variable, which
 * either matches or doesn't varnames (since orig if second order only has
 * one variable).  if it matches, we just return. otherwise, just recurse.
 */
void make_tcdeps_rec_q(struct tr *tr, struct structure *struc, struct node *orig,
		       struct node *form, char **varnames, int n_vars)
{
	struct node *varlist, *tmp, *restr, *phi;
	char **nv;
	int i, j, *hits, hit, n_nv;

	if (form->label == SOE)
	{
		if (!strcmp(varnames[0], form->l->l->l->l->data))
			return;
		restr = form->l->r;
		phi = form->r;
		make_tcdeps_rec(tr, struc, orig, phi, varnames, n_vars);
		if (restr)
			make_tcdeps_rec(tr, struc, orig, restr, varnames, n_vars);
		return;
	}

	/* otherwise we're FO */
	varlist=form->l->l;
	hit=0;
	hits = calloc(n_vars, sizeof(int));

	for (tmp=varlist; tmp; tmp=tmp->r)
		for (i=0; i<n_vars; i++)
		{
			if (!strcmp(tmp->data, varnames[i]))
			{
				hit++;
				hits[i] = 1;
				break;
			}
		}
	if (hit==0)
	{
		n_nv = n_vars;
		nv = varnames;
	}
	else
	{
		n_nv = n_vars-hit;
		nv = malloc(sizeof(char *)*(n_nv));
		for (i=0,j=0; i<n_vars; i++)
			if (hits[i]==0)
				nv[j++] = varnames[i];
	}

	restr = form->l->r;
	phi = form->r;

	make_tcdeps_rec(tr, struc, orig, phi, nv, n_nv);
	if (restr)
		make_tcdeps_rec(tr, struc, orig, restr, nv, n_nv);

	free(hits);
	if (hit==0)
		return;
	free(nv);
	return;
}

/* making tcdeps for orig, form is a TC.
 * TCs are not nested in trans.c, so get the free variables, 
 * check if any match varnames.
 * if so, add or make a tcdeps entry for orig.
 */
void make_tcdeps_rec_tc(struct tr *tr, struct structure *struc, struct node *orig,
			struct node *form, char **varnames, int n_vars)
{
	struct list *fvars = d_free_var(form, struc->vocab, 1);
	struct list *tmp, *tmpn;
	struct tcdeps *tcd=NULL, *ntcd;
	struct tcvc *tcvc=NULL;
	struct tcvc **tcp;
	int i, hit=0;

	for (tmp=fvars; tmp; tmp=tmp->next)
	{
		for (i=0; i<n_vars; i++)
			if (!strcmp(tmp->data, varnames[i]))
			{
				hit=1;
				break;
			}
		if (hit)
			break;
	}

	for (tmp=fvars; tmp; tmp=tmpn)
	{
		tmpn=tmp->next;
		free(tmp);
	}

	if (hit==0) /* nothing to do */
		return;

	/* okay, we found a dependency */
	for (i=0; i<tr->n_tcdeps; i++)
		if (tr->tcdeps[i].form==orig)
		{
			tcd = &(tr->tcdeps[i]);
			break;
		}

	if (tcd==NULL)
	{
		ntcd = realloc(tr->tcdeps, sizeof(struct tcdeps)*(tr->n_tcdeps+1));
		if (ntcd==NULL)
		{
			printf("tr26: unable to allocate memory\n");
			tr->abort=1;
			return;
		}
		tr->tcdeps = ntcd;
		tcd = &(tr->tcdeps[tr->n_tcdeps]);
		tr->n_tcdeps++;
		tcd->form = orig;
		tcd->tcvc = NULL;
		tcd->n_tcvc = 0;
	}

	/* add this TC to tcd->tcvc */
	for (i=0; i<tcd->n_tcvc; i++)
	{
		if (tcd->tcvc[i]->tcform == form)
		{
			tcvc = tcd->tcvc[i];
			break;
		}
	}

	if (tcvc==NULL) /* normal case */
	{
		tcp = realloc(tcd->tcvc, sizeof(struct tcvc *)*(tcd->n_tcvc+1));
		if (tcp==NULL)
		{
			printf("tr27: unable to allocate memory\n");
			tr->abort=1;
			return;
		}
		tcd->tcvc = tcp;
		for (i=0; i<tr->n_tcvc; i++)
			if (tr->tcvc[i].tcform == form)
			{
				tcvc = &(tr->tcvc[i]);
				break;
			}
		if (tcvc==NULL)
		{
			printf("tr28: note: couldn't find necessary tcvc\n");
			tr->abort=1;
		}
		tcd->tcvc[tcd->n_tcvc] = tcvc;
		tcd->n_tcvc++;
	}
}

/* orig is a formula binding variables varnames[0]...varnames[n_vars-1]
 * we want to search through form, looking for any TC operators.
 * if we find a TC, we want to look at it's free variables.
 * if these are bound by orig, we want to add them to orig's tcdeps,
 * or create it if relevant.
 *
 * note: if we hit another quantifier that rebinds any of these
 * variables, we remove the rebound variables when recursing -
 * since TC there can't depend on the relevant bindings in orig.
 */
void make_tcdeps_rec(struct tr *tr, struct structure *struc, 
		     struct node *orig, struct node *form,
		     char **varnames, int n_vars)
{
	if (form==NULL)
		return;

	switch (form->label)
	{
		case TRUE:
		case FALSE:
		case NUMBER:
		case LT:
		case LTE:
		case NEQUALS:
		case EQUALS:
		case MULT:
		case PLUS:
		case MINUS:
		case CONSTANT:
		case PRED:
		case VAR:
			break;
		case AND:
		case OR:
		case IMPLIES:
		case IFF:
		case XOR:
			make_tcdeps_rec(tr,struc,orig,form->l,varnames,n_vars);
			make_tcdeps_rec(tr,struc,orig,form->r,varnames,n_vars);
			break;
		case NOT:
			make_tcdeps_rec(tr,struc,orig,form->l,varnames,n_vars);
			break;
		case EXISTS:
		case FORALL:
		case SOE:
			make_tcdeps_rec_q(tr,struc,orig,form,varnames,n_vars);
			break;
		case TC:
			make_tcdeps_rec_tc(tr,struc,orig,form,varnames,n_vars);
			break;
		default:
			printf("tr29: unknown node (%d) in make_tcdeps\n",
			       form->label);
			break;
	}
}

/* TC is not allowed to be nested, so we don't need to think about that */
/* form is either a first-order or second-order quantifier formula.
 * we want to make/add a tcdeps block (if needed) in tr, which lists the
 * TCs that have free variables bound by this block.
 */
void make_tcdeps(struct tr *tr, struct structure *struc, struct node *form)
{
	struct node *tmp, *varlist, *restr=form->l->r, *phi=form->r;
	char **varnames;
	int n_vars,i;

	if (form->label == SOE)
	{
		n_vars = 1;
		varnames=malloc(sizeof(char *));
		varnames[0] = form->l->l->l->l->data;
	}
	else /* first-order */
	{
		varlist = form->l->l;
		tmp=varlist;
		n_vars=0;
		while (tmp)
		{
			n_vars++;
			tmp=tmp->r;
		}
		varnames = malloc(sizeof(char *)*n_vars);
		tmp=varlist;
		i=0;
		while (tmp)
		{
			varnames[i++] = tmp->data;
			tmp=tmp->r;
		}
	}
	make_tcdeps_rec(tr, struc, form, phi, varnames, n_vars);
	make_tcdeps_rec(tr, struc, form, restr, varnames, n_vars);

	free(varnames);
}

/* also handles tcdeps */
void make_tr_tcsovc(struct tr *tr, struct structure *struc, struct node *form)
{
	int i, hit;
	struct sovc *tmp;

	if (form==NULL)
		return;

	switch (form->label)
	{
		case TRUE:
		case FALSE:
		case LT:
		case LTE:
		case NEQUALS:
		case EQUALS:
		case PRED:
			return;
		case NOT:
			make_tr_tcsovc(tr, struc, form->l);
			return;
		case AND:
		case OR:
		case IMPLIES:
		case IFF:
		case XOR:
			make_tr_tcsovc(tr, struc, form->l);
			make_tr_tcsovc(tr, struc, form->r);
			return;
		case EXISTS:
		case FORALL:
			make_tr_tcsovc(tr, struc, form->l->r); /* restriction */
			make_tr_tcsovc(tr, struc, form->r); /* formula */
			make_tcdeps(tr, struc, form);
			return;
		case TC:
			make_tr_tcvc(tr, form); /* do the tcvc */
			make_tr_tcsovc(tr, struc, form->l->r);
			return;
		/* case IFP:  IFP not supported yet */
		case SOE:
			hit=0;
			for (i=0; i<tr->n_sovc; i++)
				if (!strcmp(tr->sovc[i].varname,
					    form->l->l->l->l->data))
				{
					hit=1;
					break;
				}
			if (hit==0) /* add the new symbol to the sovc array */
			{
				tmp = realloc(tr->sovc,
					      sizeof(struct sovc)*(tr->n_sovc+1));
				if (tmp == NULL) /* uh oh */
				{
					printf("tr7: memory error\n");
					return;
				}
				tr->sovc = tmp;
				i=tr->n_sovc++;
				tr->sovc[i].varname=
						   dupstr(form->l->l->l->l->data);
				tr->sovc[i].cur = 0;
				tr->sovc[i].next =1;
			}
			/* regardless, need to check remaining formula */
			make_tr_tcsovc(tr, struc, form->l->r);
			make_tr_tcsovc(tr, struc, form->r);
			make_tcdeps(tr, struc, form);
			return;
	}
	return;	
}

/* we want a node of type TYPE, with left and right as given (or NULL)
 * look this node up in the hash, and re-use it if possible.
 * otherwise, create it and add it to the hash
 * negl, negr are booleans for whether left/right are negated
 */
struct dagnode *tr_get_dagnode(struct tr *tr, int type, struct dagnode *left,
			       struct dagnode *right, int negl, int negr)
{
	struct hnode_t *hnode;
	char *dup;
	struct dagnode *ret;

	char cleft[32]={0};
	char cright[32]={0};
	char key[96]={0};
	char op=' ';
	int len;
	int count=0;

	if (left!=NULL)
		sprintf(cleft, "%c%d", (negl==1?'-':' '), left->num);
	if (right!=NULL)
		sprintf(cright, "%c%d", (negr==1?'-':' '), right->num);

	strcpy(key, cleft);

	if (type == TR_AND)
		op='&';
	else if (type == TR_OR)
		op='|';
	else if (type == TR_XOR)
		op='^';
	else if (type == TR_NOT)
		op='-';
	else
	{
		printf("tr8: unknown type %d\n", type);
		return NULL;
	}

	len = strlen(key);
	sprintf(key + len, "%c%s", op, cright);

	/* look for this dagnode in the hash */
	hnode = hash_lookup(tr->hash->id_hash, key);
	if (hnode != NULL)
	{
		tr_debug2("trd: hit %s in hashtable\n", key);
		ret = (struct dagnode *)hnode_get(hnode);
		tr->hits++;
		return ret;
	}
	/* TODO also check for symmetric versions ? */

	/* no hit, make one and add it */
	ret = calloc(1, sizeof(struct dagnode));
	ret->type = type;
	ret->num = tr->hash->next_id++;
	if (left!=NULL)
		count++;
	if (right!=NULL)
		count++;
	if (count>0)
	{
		ret->child = malloc(sizeof(struct dagnode *)*count);
		ret->neg = malloc(sizeof(int)*count);
	}
	ret->n_child = count;
	count = 0;
	if (left!=NULL)
	{
		ret->neg[count] = negl;
		ret->child[count++] = left;
	}
	if (right!=NULL)
	{
		ret->neg[count] = negr;
		ret->child[count++] = right;
	}

	/* insert in hash table */
	dup = dupstr(key);
	tr->keys = add_list(tr->keys, dup);
	if (!hash_alloc_insert(tr->hash->id_hash, dup, ret))
		printf("tr9: hash insert error, try increasing TR_MAXIDS\n");
	tr_debug3("trd: inserted (%s,%d) to hash\n", dup, ret->num);
	return ret;
}

/* qcir specification says these are correct, but qbf4j and qbf_mcmas don't
 * support them
 */
/* qcir true is and gate with zero inputs */
/* qcirhack: use simple quantified variable asserted to be true */
struct dagnode *tr_get_true(struct tr *tr)
{
	struct dagnode *d=&dagn_true;
	if (tr->qcirhack)
		return d;
	return tr_get_dagnode(tr, TR_AND, NULL, NULL, 0, 0);
}

/* qcir false is or gate with zero inputs */
/* qcirhack: use simple quantified variable asserted to be true */
struct dagnode *tr_get_false(struct tr *tr)
{
	struct dagnode *d=&dagn_false;
	if (tr->qcirhack)
		return d;
	return tr_get_dagnode(tr, TR_OR, NULL, NULL, 0, 0);
}

/* variadic version of tr_get_dagnode above */
/* if neg is NULL, nothing is negated */

struct dagnode *tr_get_dagnode_v(struct tr *tr, int type, 
				 struct dagnode **children, int *neg, long n)
{
	struct hnode_t *hnode;
	struct dagnode *ret;
	char *key;

	long i, len=0, offs;
	char op;

	/* if asked for a unary AND or OR of one non-negated child, just
	 * return the child, omitting the unary AND/OR (2025.3.29)
	 */
	if (n==1 && (neg==NULL || neg[0]==0) && (type==TR_AND || type==TR_OR))
		return children[0];

	/* get length for key */
	for (i=0; i<n; i++)
		len+=snprintf(NULL, 0, "%d", children[i]->num);
	len += 2*n; /* n negations, n-1 operators between, 1 closing \0 */

	key = malloc(len * sizeof(char));

	if (type == TR_AND)
		op='&';
	else if (type == TR_OR)
		op='|';
	else    /* we don't do XOR over non-binary things */
	{
		printf("tr10: unsupported variadic type %d, good luck\n", type);
		op='?'; 
	}

	offs=0;
	for (i=0; i<n; i++)
	{
		offs += sprintf(key+offs, "%c%d",
				(neg==NULL||neg[i]==0)?' ':'-',
				children[i]->num);
		if (i+1 != n)
			offs += sprintf(key+offs, "%c", op);
	}

	/* look for this in the hash */
	hnode = hash_lookup(tr->hash->id_hash, key);
	if (hnode != NULL)
	{
		tr_debug2("trd: hit %s in hashtable\n", key);
		ret = (struct dagnode *)hnode_get(hnode);
		tr->hits++;
		free(key);
		return ret;
	}

	/* no hit, make one and add it */
	ret = calloc(1, sizeof(struct dagnode));
	ret->type = type;
	ret->num = tr->hash->next_id++;

	ret->child = calloc(n, sizeof(struct dagnode *));
	ret->neg = calloc(n, sizeof(int));
	for (i=0; i<n; i++)
	{
		ret->child[i] = children[i];
		if (neg!=NULL && neg[i]==1)
			ret->neg[i] = 1;
	}
	ret->n_child = n;

	tr->keys = add_list(tr->keys, key);
	if (!hash_alloc_insert(tr->hash->id_hash, key, ret))
		printf("tr11: hash insert error, try increasing TR_MAXIDS\n");
	tr_debug3("trd: inserted (%s,%d) to hash\n", key, ret->num);
	return ret;
}


/* based on ex_p1rec_exists, which is based on eval_exists
 *
 * produced a big disjunction over the possible values of the variables.
 * for each tuple of the variables, we include a clause (restr&phi)
 * if there is a restriction and (phi) otherwise.
 */
struct dagnode *make_tr_exists(struct structure *struc, struct node *form,
			       struct tr *tr, struct interp *interp)
{
	struct node *restr = form->l->r;
	struct node *varlist = form->l->l;
	struct node *phi = form->r;
	struct node *tnode = varlist;
	struct dagnode **clauses;
	struct dagnode *tmpl, *tmpr, *ret;
	struct tcdeps *tcdeps=NULL;

	int *old_values;  /* keep old variable variables, to restore later */
	int **values;     /* pointers to values in fast interpretation */
	char **varnames;  /* names of quantified variables */

	int *first;

	int size = struc->size;
	int arity=0;
	int i, res;
	long j, n_cl=0;

	while (tnode)
	{
		arity++;
		tnode = tnode->r;
	}

	old_values = malloc(arity * sizeof(int));
	values = malloc(arity * sizeof(int *));
	varnames = malloc(arity*sizeof(char *));

	n_cl = de_pow(size, arity);
	clauses = calloc(n_cl, sizeof(struct dagnode *));
	tnode = varlist;

	for (i=0; i<arity; i++)
	{
		varnames[i] = tnode->data;
		values[i] = tnode->ival;
		old_values[i] = *(values[i]);
		*(values[i])=0;
		tnode = tnode->r;
	}

	for (i=0; i<tr->n_tcdeps; i++)
		if (tr->tcdeps[i].form==form)
		{
			tcdeps=tcdeps+i;
			break;
		}

	first = values[0];
	*first = -1;
	j=0;

	/* do next tuple */
	while (1)
	{
		/* increment nexti for all SO variables - if an
		 * SO quantifier is inside the FO quantifier, it may
		 * depend on the FO quantified variable - we use distinct
		 * QBF variables in that case to avoid confusing solvers by
		 * shadowing variables
		 */
		for (i=0; i<tr->n_sovc; i++)
			tr->sovc[i].next++;
		/* likewise for TC, the TC may depend on the variables
		 * we should track dependencies more closely, but for
		 * now we just track dependency on something in the
		 * quantifier block
		 */
		if (tcdeps!=NULL)
			for (i=0; i<tcdeps->n_tcvc; i++)
				tcdeps->tcvc[i]->cur++;

		(*first)++;
		if (*first>=size)
		{
			*first=0;
			res=0;
			for (i=1; i<arity; i++)
			{
				res = ++(*values[i]);
				if (res < size)
					break;
				res = *(values[i]) = 0;
			}
			if (!res && i==arity) /* all done */
				break;
		}
		/* make this one */
		tmpr = to_qcir_rec(struc, phi, tr, interp);
		if (restr)
		{
			tmpl = to_qcir_rec(struc, restr, tr, interp);
			clauses[j] = tr_get_dagnode(tr, TR_AND, tmpl, tmpr,0,0);
		}
		else
			clauses[j] = tmpr;
		j++;
	}

	/* restore old values */
	for (i=0; i<arity; i++)
		*(values[i]) = old_values[i];
	free(values);
	free(old_values);
	free(varnames);

	/* make a big disjunction over clauses and return it */
	ret = tr_get_dagnode_v(tr, TR_OR, clauses, NULL, n_cl);
	free(clauses);
	return ret;
}

/* based on ex_p1rec_forall
 * \forall x:\phi === !\exists x:!\phi
 */
struct dagnode *make_tr_forall(struct structure *struc, struct node *form,
			       struct tr *tr, struct interp *interp)
{
	struct node *not = node(NOT, form->r, 0);
	struct dagnode *ret;

	if (!not)
		return NULL;
	form->r = not;
	form->label = EXISTS;
	ret = make_tr_exists(struc, form, tr, interp);
	ret = tr_get_dagnode(tr, TR_AND, ret, NULL, 1, 0); /* negate */
	/* cleanup */
	form->r = not->l;
	form->label = FORALL;
	free(not);
	return ret;
}

/* given the SO variable name varname, sovc counter cur, and this tuple,
 * make the QBF variable name varname_cur_tuple[0]_tuple[1]..._tuple[arity-1]
 *
 * ideally, we'd use that... but the tools I've managed to build only support
 * integer names. so we hash the names, get an int, and return it as a
 * string (in the hope this can be avoided in future)
 *
 * this is now more difficult to change, qcir cleansing/etc in convert.c
 * also assume integer names as done here.
 */
char *tr_makevarname(struct tr *tr, const char *varname, int cur, int *tuple,
		     int arity)
{
	char *newname;
	char *ret;
	struct hnode_t *hnode;
	int olen=snprintf(NULL, 0, "%s_%d",varname,cur)+1; /* include '\0' */
	int len, i;
	int *tmp;
	int var;
	
	len=0;
	/* varname_cur_u1_u2_u3...uarity */
	for (i=0; i<arity; i++)
		len+=snprintf(NULL, 0, "_%d", tuple[i]);

	newname = malloc(sizeof(char)*(len+olen));
	len=sprintf(newname, "%s_%d",varname,cur);
	for (i=0; i<arity; i++)
		len+=sprintf(newname+len, "_%d", tuple[i]);
  /*  apparently we have to produce at least semi-cleansed format
   *  for any tools to work ...
   *  so we use the names as a key and use integers like gates.
   *  we put things in tr->keys to get freed later
   *	return newname;
   */

	hnode = hash_lookup(tr->hash->id_hash, newname);
	if (hnode != NULL)
	{
		tr_debug2("trd: hit %s in hashtable\n", newname);
		var = *(int *)hnode_get(hnode);
		tr->hits++;
		free(newname);
	}
	else
	{
		/* make a new one and insert it */
		tmp=malloc(sizeof(int));
		*tmp=tr->hash->next_id++;
		if (!hash_alloc_insert(tr->hash->id_hash, newname, tmp))
			printf("tr12: hash insert error, try increasing TR_MAXIDS\n");
		tr->keys = add_list(tr->keys, tmp);
		tr->keys = add_list(tr->keys, newname);
		tr_debug3("trd: inserted (%s,%d) to hash\n", newname, *tmp);
		var=*tmp;
	}

	len = snprintf(NULL, 0, "%d", var);
	ret = malloc(sizeof(char)*(len+1));
	sprintf(ret, "%d", var);
	return ret;
}

struct sovc *tr_get_sovc(struct tr *tr, const char *varname)
{
	int i;
	for (i=0; i<tr->n_sovc; i++)
		if (!strcmp(varname, tr->sovc[i].varname))
			return tr->sovc+i;
	return NULL;
}

/* form is an SOE formula.  We need to:
 * (0) if this varname is being renamed, temporarily remove the renaming
 * (1) set the corresponding sovc entry's cur=nexti++
 * (2) make a dagnode with QBF variables for the SO variable, and body etc
 * (3) restore the sovc entry's cur to the old variable
 * (4) restore the renaming if it was removed.
 *
 * note we don't restore nexti -- there could be another SO quantifier
 * e.g. \exists R:psi \land \forall R:phi, and we want to keep
 * the QBF variable names distinct.
 *
 * also need to increment all next for other sovc,tcvc to avoid hitting
 * hashtable entries that are no longer relevant
 */
struct dagnode *make_tr_soe(struct structure *struc, struct node *form,
			    struct tr *tr, struct interp *interp)
{
	struct dagnode *ret, *dn, *left, *right;
	char *varname=form->l->l->l->l->data, *newname;
	struct relation *rel; /* insert in interp to hide symbols in vocab */
	struct relrename *rr;
	int *tuple;
	char *orel;
	int size=struc->size;
	int arity = *(int *)(form->l->l->l->r->data);
	int j, k;
	unsigned int i;
	unsigned long oldcur;
	unsigned long cur;
	unsigned long n_vars;

	struct node *restr=form->l->r;
	struct node *phi = form->r;

	struct sovc *sovc=tr_get_sovc(tr, varname);

	if (sovc==NULL)
	{
		printf("tr13: SO variable not in sovc, good luck\n");
		return NULL;
	}

	/* remove a rename if present.  at most one renaming per symbol
	 * exists
	 */
	for (rr=tr->renames; rr; rr=rr->next)
	{
		if (!strcmp(rr->oldname, varname))
		{
			tr_debug2("make_tr_soe: removing renaming of %s\n",
				  varname);
			orel = rr->newname;
			rr->newname = NULL;
		}
	}
	
	/* add a dummy SOE relation to the interpretation */
	/* so we can correctly handle SO variables that shadow
	 * predicates in the vocabulary
	 */
	rel = malloc(sizeof(struct relation));
	rel->name = varname;
	rel->arity=arity;
	rel->next=interp->rel_symbols;
	rel->cache = NULL;
	rel->parse_cache = 0;
	interp->rel_symbols = rel;

	oldcur=sovc->cur;
	sovc->cur=sovc->next; /* sovc->next incremented below */
	cur=sovc->cur;

	n_vars = de_pow(size, arity);

	/* increment all sovc next entries */
	for (j=0; j<tr->n_sovc; j++)
		tr->sovc[j].next++;

	/* increment cur values on TC depending on us */
	for (j=0; j<tr->n_tcdeps; j++)
		if (tr->tcdeps[j].form == form)
		{
			for (k=0; k<tr->tcdeps[j].n_tcvc; k++)
				tr->tcdeps[j].tcvc[k]->cur++;
			break;
		}

	/* make body of formula */
	right = to_qcir_rec(struc, phi, tr, interp);
	if (restr)
	{
		left = to_qcir_rec(struc, restr, tr, interp);
		dn = tr_get_dagnode(tr, TR_AND, left, right, 0, 0);
	}
	else
		dn = right;

	/* make ret, a dagnode of type TR_EXISTS with the appropriate
	 * body and quantified variables
	 */
	ret = calloc(1, sizeof(struct dagnode));
	ret->type = TR_EXISTS;
	ret->num = tr->hash->next_id++;
	ret->child = malloc(sizeof(struct dagnode *));
	ret->child[0] = dn;
	ret->n_child = 1;
	ret->n_vars = n_vars;
	ret->vars = malloc(sizeof(char *)*n_vars);

	tuple = NULL;

	for (i=0; i<n_vars; i++)
	{
		tuple = next_tuple(tuple, arity, size);
		newname = tr_makevarname(tr, varname, cur, tuple, arity);

		ret->vars[i] = newname;
		tr_debug2("make_tr_soe: added QBF variable %s\n", newname);
	}

	free(tuple);
	sovc->cur = oldcur; /* restore old value, going out of scope of this q*/
	interp->rel_symbols=rel->next; /* remove symbol, going out of scope */
	if (rr)
		rr->newname = orel;
	free(rel);
	return ret;
}

/*
 * three cases: predicate symbol is in vocabulary, not quantified
 *		predicate symbol is bound by SO quantifier.
 *		predicate symbol is a internal TC predicate
 *
 * needs some care: if predicate symbol is in structure, we may
 * have only a symbolic definition that needs to be evaluated.
 * but we can just use eval()/eval_pred() to get the truth value.
 *
 * needs more care: if inside a TC, we need to look at the relrenames,
 * since we will likely be referring to the predicate by a different
 * name.  TC also hacks in SO-like predicates, but they don't have
 * sovc entries - always use 0 for cur for them.
 */
struct dagnode *make_tr_pred(struct structure *struc, struct node *form,
			     struct tr *tr, struct interp *interp)
{
	struct relation *rel;
	struct dagnode *ret;
	char *relname = (char *)form->data, *varname;
	char *oldname;
	struct sovc *sovc;
	struct node *relargs;
	int *tuple;
	struct relrename *rr;
	int arity, i, cur, tcflag=0;
	int res;

	for (rr=tr->renames; rr; rr=rr->next)
	{ /* check if we're renaming this predicate because of inside a TC*/
		if (!strcmp(relname, rr->oldname) && rr->newname!=NULL)
		{
			tcflag = 1;
			relname = rr->newname;
			break;
		}
	}
	/* check in interpretation for most recent SOE binding */
	rel = get_relation(relname, interp, NULL);

	if (rel)
	{	/* based on eval_pred */
		arity = rel->arity;
		tuple = malloc(sizeof(int)*arity);
		relargs = form->r;
		for (i=0; relargs && i<arity; i++)
		{
			tuple[i] = teval(relargs->l, interp, struc);
			if (tuple[i]>=struc->size || tuple[i]<0)
			{
				free(tuple);
				return tr_get_false(tr); /* out of range = false */
			}
			relargs = relargs->r;
		}
		if (relargs || i!=arity)
		{
			printf("tr14: second-order variable %s used with incorrect arity\n", relname);
			free(tuple);
			return tr_get_false(tr);
		}

		sovc = tr_get_sovc(tr, relname);
		if (sovc==NULL && tcflag!=1)
		{
			printf("tr15: can't find sovc for %s, good luck\n", relname);
			cur = 0;
		}
		else if (sovc==NULL && tcflag==1)
			cur = 0;
		else
			cur = sovc->cur;
		varname = tr_makevarname(tr, relname, cur, tuple, arity);

		free(tuple);
		ret = calloc(1, sizeof(struct dagnode));
		ret->type = TR_VAR;
		ret->num = tr->hash->next_id++;
		ret->child = NULL;
		ret->n_child = 0;
		ret->vars = malloc(sizeof(char *));
		ret->vars[0] = varname;
		ret->n_vars = 1;
		ret->neg = malloc(sizeof(int));
		ret->neg[0] = 0;
		ret->printed = 0;
		return ret;
	}
	else if ( (rel = get_relation(relname, NULL, struc)) )
	{
		/* 2025.3.29: if we've renamed relname above, need
		 * to stick the renamed version in relname for eval_pred
		 * to find !!!
		 */
		if (tcflag)
		{
			oldname = form->data;
			form->data=relname;
		}
		/* relation not SO bound, but in structure */
		res = eval_pred(form, interp, struc);

		if (tcflag)
			form->data=oldname;

		if (res == 0) /* FALSE */
			return tr_get_false(tr);
		else if (res == 1) /* TRUE */
			return tr_get_true(tr);
		/* otherwise it's a confused formula */
		printf("tr16: broken predicate %s\n", relname);
		tr->abort = 1;
		return NULL;
	}
	else
	{
		printf("tr17: unknown predicate %s\n", relname);
		tr->abort = 1;
		return NULL;
	}
}

/* return _TCtcnum_cur_distance_u1_..._ua_v1_..._va
 * the hash key (varname) for whether there is a path of distance
 * between tup1 (u1...ua) and tup2 (v1...va)
 * for predicate variables, replace ui with Ri for symbols in the vocabulary
 * and Ri%d where %d is the cur from sovc for Ri (when a second-order or
 * TC-introduced second-order-like symbol)
 */
char *tr_get_tckey(struct tr *tr, struct tcvc *tcvc,unsigned long long distance,
		   int *tup1, char **rel1, int *tup2, char **rel2,
		   struct interp *interp)
{
	struct relation *rel;
	struct sovc *sovc;
	char *ret;
	int *cur1, *cur2;
	int *sorts = tcvc->sorts;
	int i, arity=tcvc->tup_arity;
	unsigned long len, offs;

	cur1 = calloc(arity, sizeof(int));
	cur2 = calloc(arity, sizeof(int));

	len = snprintf(NULL, 0,"_TC%d_%lu_%llu",tcvc->num,tcvc->cur,distance)+1;

	for (i=0; i<arity; i++)
	{
		if (sorts[i]==-1)
			len += snprintf(NULL, 0, "_%d", tup1[i]) + 
			       snprintf(NULL, 0, "_%d", tup2[i]);
		else
		{
			rel = get_relation(rel1[i], interp, NULL);
			if (rel) /* second-order variable, need sovc cur value*/
			{
				sovc = tr_get_sovc(tr, rel1[i]);
				if (sovc!=NULL) /* cur1[i] init to 0 by calloc*/
					cur1[i]=sovc->cur;
				len += snprintf(NULL, 0, "_%s-%d", rel1[i],
						cur1[i]);
			}
			else /* vocabulary symbol, no need to use cur */
			{
				cur1[i]=-1;
				len += snprintf(NULL, 0, "_%s", rel1[i]);
			}

			rel = get_relation(rel2[i], interp, NULL);
			if (rel) /* second-order variable, need sovc cur value*/
			{
				sovc = tr_get_sovc(tr, rel2[i]);
				if (sovc!=NULL) /* cur2[i] init to 0 by calloc */
					cur2[i] = sovc->cur;
				len += snprintf(NULL, 0, "_%s-%d", rel2[i],
						cur2[i]);
			}
			else /* vocabulary symbol, no need to use cur */
			{
				cur2[i] = -1;
				len += snprintf(NULL, 0, "_%s", rel2[i]);
			}
		}
	}
	ret = malloc(len*sizeof(char));

	offs=sprintf(ret, "_TC%d_%lu_%llu", tcvc->num, tcvc->cur, distance);
	for (i=0; i<arity; i++)
	{
		if (sorts[i] == -1)
			offs+=sprintf(ret+offs, "_%d", tup1[i]);
		else if (cur1[i]>=0) /* second order variable */
			offs+=sprintf(ret+offs, "_%s-%d", rel1[i], cur1[i]);
		else /* vocabulary symbol, no need to use cur */
			offs+=sprintf(ret+offs, "_%s", rel1[i]);
	}

	for (i=0; i<arity; i++)
	{
		if (sorts[i] == -1)
			offs+=sprintf(ret+offs, "_%d", tup2[i]);
		else if (cur2[i]>=0) /* second order variable */
			offs+=sprintf(ret+offs, "_%s-%d", rel2[i], cur2[i]);
		else /* vocabulary symbol, no need to use cur */
			offs+=sprintf(ret+offs, "_%s", rel2[i]);
	}

	tr_debug2("trd: got tckey %s\n", ret);
	free(cur1);
	free(cur2);
	return ret;
}

/* make a dagnode for whether the tcform is true between tup1/rel1 and
 * tup2/rel2.
 * distance is 1, tup1!=tup2 and it's not in the cache
 * for now we forbid nested TC, which simplifies renaming somewhat.
 */
struct dagnode *make_tr_tc1dagnode(struct structure *struc,struct node *form,
				   struct tr *tr, struct tcvc *tcvc,
				   struct interp *interp, int *tup1,
				   char **rel1, int *tup2, char **rel2)
{
	int *old_values1, *old_values2;
	int **values1=tcvc->values1, **values2=tcvc->values2;
	struct node *tcform=form->l->r;
	struct relrename **renames1=tcvc->renames1,**renames2=tcvc->renames2;
	struct dagnode *ret;
	int *sorts = tcvc->sorts, tup_arity = tcvc->tup_arity;
	char **orel1, **orel2;
	int i;

	old_values1=malloc(sizeof(int)*tup_arity);
	old_values2=malloc(sizeof(int)*tup_arity);
	orel1 = malloc(sizeof(char *)*tup_arity);
	orel2 = malloc(sizeof(char *)*tup_arity);

	/* get old variable values */
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
		{
			old_values1[i] = *(values1[i]);
			old_values2[i] = *(values2[i]);
		}
		else 
		{
			orel1[i] = renames1[i]->newname;
			orel2[i] = renames2[i]->newname;
		}
	}
	
	/* add values of variables to interpretation */
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
		{
			*(values1[i]) = tup1[i];
			*(values2[i]) = tup2[i];
		}
		else
		{
			renames1[i]->newname = rel1[i];
			renames2[i]->newname = rel2[i];
		}
	}

	/* any SO or TC nested inside can depend on these tuples,
	 * so need to increment next values
	 */
	/* we don't need to increment any TC predicate variables
	 * associated with the current TC, but just do them all
	 */
	for (i=0; i<tr->n_sovc; i++)
		tr->sovc[i].next++;

	/* recurse */
	ret = to_qcir_rec(struc, tcform, tr, interp);

	/* remove values of variables from interpretation */
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
		{
			*(values1[i]) = old_values1[i];
			*(values2[i]) = old_values2[i];
		}
		else
		{
			renames1[i]->newname = orel1[i];
			renames2[i]->newname = orel2[i];
		}
	}

	/* clean memory */
	free(old_values1);
	free(old_values2);
	free(orel1);
	free(orel2);

	return ret;
}

/* make this TC dagnode, we know it's not in the hash and that distance>1
 * path of distance betwen tup1/rel1 and tup2/rel2, iff exists some tup3/rel3
 * st path of distance/2 between tup1 and tup3 and also between tup3 and tup2
 * note we're doing reflexive TC, but this is done by having a path of
 * all lengths between tup and itself.
 * if this is not a TC1, we use SOE quantifiers for the intermediate rel3
 */
struct dagnode *make_tr_rec_tcdagnode(struct structure *struc,struct node *form,
				      struct tr *tr, struct interp *interp,
				      struct tcvc *tcvc, unsigned long distance,
				      int *tup1, char **rel1, int *tup2,
				      char **rel2)
{
	struct dagnode *ret =calloc(1,sizeof(struct dagnode)),*tmp,*tmp1,*tmp2;
	int *tup3=NULL, *tuple=NULL;
	int *sorts = tcvc->sorts;
	char **rel3;
	struct relation *rel;
	int i, j, n_fo=0, n_so=0;
	unsigned long ui, size=struc->size;
	int tup_arity = tcvc->tup_arity;
	unsigned long num_tuples, n_vars;
	unsigned long long d2 = distance>>1;
	unsigned long len;

	/* set up rel3 */
	/* we use always distinct predicate names that can't
	 * be obtained by the user - so we don't need to use cur/next
	 * values or sovc entries
	 */
	rel3 = calloc(tup_arity, sizeof(char *));
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]!=-1) /* found a predicate sort */
		{
			n_so++;
			len = snprintf(NULL, 0, "_T%d_%lu", tcvc->num,
				       tcvc->counter);
			rel3[i] = malloc(sizeof(char)*(len+1));
			sprintf(rel3[i], "_T%d_%lu", tcvc->num,
				tcvc->counter++);
			rel = malloc(sizeof(struct relation));
			rel->name = rel3[i];
			rel->arity = sorts[i];
			rel->next = interp->rel_symbols;
			rel->cache = NULL;
			rel->parse_cache = NULL;
			interp->rel_symbols = rel;
		}
		else
			n_fo++;
	}

	num_tuples = de_pow(size, n_fo);

	ret->type = TR_OR;
	ret->num = tr->hash->next_id++;
	ret->n_child = num_tuples;
	ret->child = malloc(sizeof(struct dagnode *)*num_tuples);
	ret->vars=NULL;
	ret->n_vars=0;
	ret->neg = calloc(num_tuples, sizeof(int));
	ret->printed=0;

	tup3 = calloc(tup_arity, sizeof(int));
	for (ui=0; ui<num_tuples; ui++)
	{
		tuple = next_tuple(tuple, n_fo, size);
		for (i=0,j=0; j<tup_arity; j++)
			if (sorts[j]==-1)
				tup3[j]=tuple[i++];
		tmp1 = old_get_tr_tcdagnode(struc, form, tr, interp, tcvc,
					d2, tup1, rel1, tup3, rel3);
		tmp2 = old_get_tr_tcdagnode(struc, form, tr, interp, tcvc,
					d2, tup3, rel3, tup2, rel2);
		ret->child[ui] = tr_get_dagnode(tr, TR_AND, tmp1, tmp2, 0, 0);
	}

	tuple = next_tuple(tuple, n_fo, size); /* need to free after last */
	free(tup3);

	if (n_so>0)
	{
		n_vars = 0;
		for (i=0; i<tup_arity; i++)
			if (sorts[i]!=-1)
				n_vars += de_pow(size, sorts[i]);
		tmp = calloc(1, sizeof(struct dagnode));
		tmp->type = TR_EXISTS;
		tmp->num = tr->hash->next_id++;
		tmp->child = malloc(sizeof(struct dagnode *));
		tmp->child[0] = ret;
		tmp->n_child = 1;
		tmp->n_vars = n_vars;
		tmp->vars = malloc(sizeof(char *)*n_vars);

		j=0;
		for (i=0; i<tup_arity; i++)
		{
			if (sorts[i]==-1)
				continue;
			tuple = NULL;
			while ( (tuple=next_tuple(tuple, sorts[i], size) ))
			{
				tmp->vars[j++]=tr_makevarname(tr,rel3[i],0,
							     tuple,sorts[i]);
			
			}
		}
		ret = tmp;
	}

	/* remove things added to interp */
	for (i=0; i<n_so; i++)
	{
		rel = interp->rel_symbols;
		interp->rel_symbols = rel->next;
		free(rel);
	}
	for (i=0; i<tup_arity; i++)
		free(rel3[i]);
	free(rel3);

	return ret;
}

/* make a new cache for the relation, with given n and arity,
 * initially no values known
 */
int *make_new_cache(int n, int arity)
{
	unsigned long i, cache_size = de_pow(n, arity);
	int *cache = malloc(cache_size*sizeof(int));

	for (i=0; i<cache_size; i++)
		cache[i] = -1;

	return cache;
}

/* returns 1 if the relations are different, 0 if the same.
 * we may need to evaluate things...
 * allocates and updates relation's cache - will be a full cache if
 * the predicates are the same.
 */
int tr_rel_diff(struct structure *struc, struct interp *interp,
		struct relation *rel1, struct relation *rel2, int n)
{
	int *tuple=NULL, arity=rel1->arity, res1, res2, ret=0, index=-1;
	int *old_values, **values, i;

	if (arity!=rel2->arity) /* that really shouldn't happen */
		return 1;	/* but they're different */

	old_values = malloc(sizeof(int)*arity);
	values = malloc(sizeof(int *)*arity);

	for (i=0; i<arity; i++)
	{
		values[i] = get_xi_ival(i+1, interp);
		old_values[i] = *(values[i]);
	}

	while ((tuple=next_tuple(tuple, arity, n)))
	{
		index++;
		interp = add_tup_to_interp(interp, tuple, arity);
		/* evaluate in rel1 */
		if (rel1->cache)
		{
			res1=rel1->cache[index];
			if (res1<0)
			{
				for (i=0; i<arity; i++)
					*(values[i]) = tuple[i];
				res1 = eval_rec(rel1->parse_cache, interp, struc);
			}
		}
		else /* rel1->cache==NULL */
		{
			for (i=0; i<arity; i++)
				*(values[i])=tuple[i];
			res1 = eval_rec(rel1->parse_cache, interp, struc);
		}
		/* evaluate in rel2 */
		if (rel2->cache)
		{
			res2=rel2->cache[index];
			if (res2<0)
			{
				for (i=0; i<arity; i++)
					*(values[i]) = tuple[i];
				res2 = eval_rec(rel2->parse_cache, interp, struc);
			}
		}
		else /* rel2->cache==NULL */
		{
			for (i=0; i<arity; i++)
				*(values[i]) = tuple[i];
			res2 = eval_rec(rel2->parse_cache, interp, struc);
		}

		/* if not in cache, allocate and update the cache */
		if (rel1->cache==NULL)
			rel1->cache = make_new_cache(n, arity);
		if (rel2->cache==NULL)
			rel2->cache = make_new_cache(n, arity);
		if (rel1->cache[index]<0)
			rel1->cache[index] = res1;
		if (rel2->cache[index]<0)
			rel2->cache[index] = res2;

		if (res1 != res2)
		{
			ret = 1;
			break;
		}
	}

	/* replace old values */
	for (i=0; i<arity; i++)
		*(values[i]) = old_values[i];

	free(old_values);
	free(values);

	return ret;
}

/* checks whether tuples tup1(rel1) and tup2(rel2) are the same tuple.
 * used for checking reflexive transitive closure.
 *
 * returns:
 *  0 if different, 1 if the same, 2 if can't be determined
 *
 * (ie 2 if at least one of the relations is a quantified SO/TC predicate
 * that may or may not be equal to the corresponding entry
 */ 
int tr_check_equal_tup(struct structure *struc, struct interp *interp, int *tup1,
		       int *tup2, char **rel1, char **rel2, int tup_arity, 
		       int *sorts)
{
	struct relation *rrel1, *rrel2;
	int i, ret = 1;

	/* ret is initialized to 1, never changed back to 1. if changed to 0,
 	 * we immediately break and return.
	 * if some elements are quantified predicates, we may still be able to
	 * determine the tuples are different based on other elements -
	 * avoiding 'return 2' is helpful for producing smaller formulas
	 */
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]!=-1 && (get_relation(rel1[i], interp, NULL) ||
				     get_relation(rel2[i], interp, NULL)) )
		{
			ret = 2;
			continue;
		}
		/* compare element i */
		else if (sorts[i] == -1) /* lucky us, we can just compare it */
		{
			if (tup1[i]!=tup2[i])
			{
				ret=0;
				break;
			}
		}
		else
		{
			rrel1 = get_relation(rel1[i], NULL, struc);
			rrel2 = get_relation(rel2[i], NULL, struc);
			if (tr_rel_diff(struc, interp,rrel1,rrel2, struc->size))
			{
				ret=0;
				break;
			}
		}
	}
	return ret;
}

/* return a dagnode with this variable */
struct dagnode *make_tr_varnode(struct tr *tr, char *varname)
{
	struct dagnode *ret = calloc(1, sizeof(struct dagnode));

	ret->type = TR_VAR;
	ret->num = tr->hash->next_id++;
	ret->child = NULL;
	ret->n_child = 0;
	ret->vars = malloc(sizeof(char *));
	ret->vars[0] = varname;
	ret->n_vars = 1;
	ret->neg = malloc(sizeof(int));
	ret->neg[0] = 0;
	ret->printed = 0;

	return ret;
}

/* return a formula equivalent to "tuple1/rel1 == tuple2/rel2"
 * we've already checked that we need such a formula (ie they're not
 * obviously equal or different) so it suffices to consider only
 * entries in which at least one side is a quantified/TC predicate
 *
 * the corresponding entries can also be quantified/TC predicates,
 * or they may be defined predicates in the structure
 */
struct dagnode *tr_make_equal_check(struct tr *tr, struct structure *struc,
				    struct interp *interp,char **rel1,
				    char **rel2, int tup_arity,
				    int *sorts)
{
	struct dagnode *ret=NULL, **child, **tmp, *tmp1, *tmp2;
	char **crel, **vrel;
	struct sovc *sovc;
	struct relation *rel;
	int *neg;
	int *old_values, **values;
	int k, i, num=0, arity, n=struc->size, *tuple=NULL;
	int qr1, qr2; /* flags for whether quantified or not */
	int cur, cur1, cur2, res;
	char *vn, *vn1, *vn2;
	unsigned long j, relsize;

	for (i=0; i<tup_arity; i++) /* count entries to make formulas for */
	{
		if (sorts[i]==-1)
			continue;
		else if (get_relation(rel1[i], interp, NULL) || 
			 get_relation(rel2[i], interp, NULL))
				num++;
	}

	child = malloc(sizeof(struct dagnode *)*num);
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
			continue;
		qr1 = (get_relation(rel1[i], interp, NULL)!=NULL);
		qr2 = (get_relation(rel2[i], interp, NULL)!=NULL);
		if (!qr1 && !qr2)
			continue;
		/* make child[j] */
		arity=sorts[i];
		relsize = de_pow(n, arity);
		tmp = malloc(sizeof(struct dagnode *) * relsize);
		j=0;

		cur = -1;
		cur1 = -1;
		cur2 = -1;

		if (qr1 && qr2)
		{
			sovc = tr_get_sovc(tr, rel1[i]);
			if (sovc)
				cur1=sovc->cur;
			else
				cur1=0;
			sovc=tr_get_sovc(tr, rel2[i]);
			if (sovc)
				cur2=sovc->cur;
			else
				cur2=0;
		}
		else if (qr1)
		{
			sovc = tr_get_sovc(tr, rel1[i]);
			if (sovc)
				cur=sovc->cur;
			else
				cur=0;
			vrel = rel1;
			crel = rel2;
			rel = get_relation(crel[i], NULL, struc);
		}
		else if (qr2)
		{
			sovc = tr_get_sovc(tr, rel2[i]);
			if (sovc)
				cur=sovc->cur;
			else
				cur=0;
			crel = rel1;
			vrel = rel2;
			rel = get_relation(crel[i], NULL, struc);
		}

		j=0;

		old_values = malloc(sizeof(int)*arity);
		values = malloc(sizeof(int *)*arity);
		if (!qr1 || !qr2)
			for (k=0; k<arity; k++)
			{
				values[k] = get_xi_ival(k+1, interp);
				old_values[k] = *(values[k]);
			}

		while ( (tuple=next_tuple(tuple, arity, n)) )
		{
			/* make formula that rel1(tuple) == rel2(tuple) */
			
			/* both are quantified/TC predicates */
			if (qr1 && qr2)
			{
				vn1=tr_makevarname(tr,rel1[i],cur1,tuple,arity);
				vn2=tr_makevarname(tr,rel2[i],cur2,tuple,arity);
				tmp1 = make_tr_varnode(tr, vn1);
				tmp2 = make_tr_varnode(tr, vn2);
				tmp[j++] = tr_get_dagnode(tr, TR_XOR, tmp1, tmp2,
							  0, 1); /* xor for iff */
				continue;
			}
			else
			{
				vn = tr_makevarname(tr,vrel[i],cur,tuple,arity);
				tmp1 = make_tr_varnode(tr, vn);
				/* need to evaluate crel (relation rel) here */
				if (rel->cache==NULL || (res=rel->cache[j])<0)
				{ /* not cached */
					for (k=0; k<arity; k++)
						*(values[k]) = tuple[k];
					res = eval_rec(rel->parse_cache, interp,
						       struc);
					if (rel->cache==NULL)
						rel->cache = make_new_cache(n,
									   arity);
					rel->cache[j] = res;
				}
				if (res)
					tmp[j++] = tmp1;
				else
					tmp[j++]=tr_get_dagnode(tr, TR_NOT,tmp1,
								NULL, 0, 0);
			}

		}

		if (!qr1 || !qr2)
			for (k=0; k<arity; k++)
				*(values[k]) = old_values[k];
		free(values);
		free(old_values);

		neg = calloc(relsize, sizeof(int));
		child[i] = tr_get_dagnode_v(tr, TR_AND, tmp, neg, relsize);
		free(tmp);
		free(neg);
	}

	neg = calloc(num, sizeof(int));
	ret = tr_get_dagnode_v(tr, TR_AND, child, neg, num);
	free(child);
	free(neg);

	return ret;
}


/* we want to make an intermediate TC dagnode, representing whether
 * there is a patch from tup1 to tup2 of distance<=distance
 * this is done recursively, cutting distance in half each time,
 * with distance==1 as base case.  note this is REFLEXIVE transitive
 * closure, so if tup1==tup2, return TRUE
 *
 * sorts[] gives the sort of arguments. if a predicate variable,
 * relX[i] gives the name. if a first-order variable, tupX[i] gives the value
 */
struct dagnode *old_get_tr_tcdagnode(struct structure *struc, struct node *form,
				 struct tr *tr, struct interp *interp,
				 struct tcvc *tcvc, unsigned long long distance,
			       	 int *tup1, char **rel1, int *tup2, char **rel2)
{
	struct dagnode *ret=NULL, *body=NULL, *eq=NULL;
	struct hnode_t *hnode;
	char *key;
	int *sorts = tcvc->sorts;
	int tup_arity = tcvc->tup_arity;
	int check;

	key = tr_get_tckey(tr, tcvc, distance, tup1, rel1, tup2, rel2, interp);

	/* look for this node in the hash */
	hnode = hash_lookup(tr->hash->id_hash, key);

	/* return it from the hash */
	if (hnode!=NULL)
	{
		ret = (struct dagnode*)hnode_get(hnode);
		tr_debug2("trd: hit %s in hash, returning it\n", key);
		tr->hits++;
		free(key);
		return ret;
	}

	check=tr_check_equal_tup(struc, interp, tup1, tup2, rel1, rel2, tup_arity,
				 sorts);
	if (check==1) /* equal tuples, reflexive TC so always true */
		ret = tr_get_true(tr);/* reflexive TC so true at any distance */
	else if (check==0 && distance==0) /* definitely not equal, distance 0 */
		ret = tr_get_false(tr); /* TC false at distance 0 unless equal*/
	else if (check==2 && distance==0) /* as previous case */
		ret = tr_make_equal_check(tr, struc, interp, rel1,
					 rel2, tup_arity, sorts);
	else if (check==2) /* need to make a formula to check if equal */
		eq = tr_make_equal_check(tr, struc, interp, rel1, 
					 rel2, tup_arity, sorts);

	if (ret == NULL)
	{
		if (distance == 1) /* base case */
			body = make_tr_tc1dagnode(struc,form,tr,tcvc,interp,
						  tup1,rel1,tup2,rel2);
		else /* make recursively */
			body =make_tr_rec_tcdagnode(struc,form,tr,interp,tcvc,
						     distance,tup1,rel1,tup2,
						     rel2);
	
		if (eq) /* add recursive bit */
			ret = tr_get_dagnode(tr, TR_OR, eq, body, 0, 0);
		else
			ret = body;
	}
	/* add to hash */
	hash_alloc_insert(tr->hash->id_hash, key, ret);
	tr->keys = add_list(tr->keys, key);

	return ret;
}


/*   this is the OLD make_tr_tc - exponential if not tc1 and
 *   only somewhat tested for non-tc1.
 *   if not tc1, probably best to avoid...
 * form is a TC, ie TC[vars : phi](a,b).
 * return a dagnode representing this.
 * done by using the definition of TC in the implicit graph defined
 * by phi, creating dagnodes that represent its bits, and asserting the
 * bit corresponding to (a,b).
 * somewhat-involved.
 *
 * note : we need to handle the sovc,tcvc values.
 * all other sovc: next++
 *
 * this assumes the TC has passed the syntactic checks (matching sorts etc)
 * so relargs and second half of tcargs match first half, etc.
 */
struct dagnode *old_make_tr_tc(struct structure *struc, struct node *form,
			       struct tr *tr, struct interp *interp)

{
	struct node *tcargs=form->l->l, *relargs=form->r, *tmp;
	struct dagnode *ret;
	int *tup1, *tup2;
	char **rel1=NULL, **rel2=NULL;
	int *sorts;
	int tup_arity, size;
	int k;
	unsigned long num_args, i, r;
	unsigned long long distance, d;
	struct tcvc *tcvc=NULL;
	int j, n_tcvc=tr->n_tcvc;

	size=struc->size;

	for (tup_arity=0, tmp=tcargs; tmp; tup_arity++)
		tmp=tmp->r;
	for (i=0, tmp=relargs; tmp; i++)
		tmp=tmp->r;
	num_args = tup_arity<<1; /* parser does it with qnode */
	if (i!=num_args)
	{
		printf("tr18: TC of arity %lu given %lu arguments\n",num_args,i);
		tr->abort=1;
		return NULL;
	}

	for (j=0; j<n_tcvc; j++)
		if (form == tr->tcvc[j].tcform)
		{
			tcvc = tr->tcvc+j;
			break;
		}
	if (tcvc==NULL)
	{
		printf("tr19: TC formula not in tcvc, good luck\n");
		tr->abort=1;
		return NULL;
	}

	sorts = get_sorts(form);
	k=0;
	for (j=0; j<tup_arity; j++)
		if (sorts[j]>0)
			k++;

	for (j=0; j<tr->n_sovc; j++)
		tr->sovc[j].next++;

	tup1 = malloc(sizeof(int)*tup_arity);
	tup2 = malloc(sizeof(int)*tup_arity);
	rel1 = malloc(sizeof(char *)*tup_arity);
	rel2 = malloc(sizeof(char *)*tup_arity);
	tmp = relargs;

	for (j=0; j<tup_arity; j++,tmp=tmp->r)
	{
		if (sorts[j]==-1)
		{
			tup1[j] = teval(tmp->l, interp, struc);
			rel1[j] = NULL;
		}
		else
			rel1[j] = tmp->l->data;
	}
	for (j=0; j<tup_arity; j++,tmp=tmp->r)
	{
		if (sorts[j]==-1)
		{
			tup2[j] = teval(tmp->l, interp, struc);
			rel2[j] = NULL;
		}
		else
			rel2[j] = tmp->l->data;
	}
	/* maximum distance in the implicit graph is
	 *  D = n^r \Pi_{i=1}^k 2^{n^{a_i}} -1
	 * if there are r first-order variables, and k predicate variables
	 * per tuple.
	 *
	 * TC halfs distance at each step, so smallest power of 2
	 * larger than that is convenient.  Unfortunately, this is exponential
	 * sized if it's not TC1.  We need a new encoding.
	 *
	 * log2 uses doubles, but size (n) will be small enough to
	 * be represented exactly.
	 */

	d=1, r=0;

	for (j=0; j<tup_arity; j++)
	{
		if (sorts[j] == -1)
			r++;
		else
			d *= de_pow(2,de_pow(size, sorts[j]));
	}
	d *= de_pow(size, r);

	
	for (distance=1; distance<d-1; distance = distance<<1)
		; /* empty loop, semicolon on new line to silence warning */
	ret = old_get_tr_tcdagnode(struc, form, tr, interp, tcvc, distance,
			       tup1, rel1, tup2, rel2);
	free(tup1);
	free(tup2);
	free(sorts);
	free(rel1);
	free(rel2);
	return ret;
}

struct relation *tr_insert_rel(struct relation *next, char *name, int arity)
{
	struct relation *rel = malloc(sizeof(struct relation));
	rel->name = name;
	rel->arity = arity;
	rel->next = next;
	rel->cache = NULL;
	rel->parse_cache = NULL;
	return rel;
}

/* \psi_d(a,b) :=
 * \exists mid\forall one,two: { 
 *       ((one=a & mid=two)|(one=mid & b=two)) -> \psi_{d/2}(one,two)
 *   }
 */
/* for now we only support predicate variables (TC2) */
/* we use unique names unobtainable by the user, so no need to worry about
 * rebinding/etc names
 */
struct dagnode *new_get_tr_tcdagnode(struct structure *struc, struct node *form,
				     struct tr *tr, struct interp *interp,
				     struct tcvc *tcvc, 
				     unsigned long long distance,
				     int *tup1, char **rel1, int *tup2,
				     char **rel2, int *index)
{
	struct dagnode *ret, *tmp, *tmp1, *tmp2, *tmp3, *psi, *body, *condition,
		       *eq=NULL;
	int *sorts = tcvc->sorts, *tuple;
	char **m_rels, **one_rels, **two_rels, **m_vars, **t_vars, *key;
	struct hnode_t *hnode;
	struct relation *rel;
	int tup_arity = tcvc->tup_arity, k=0, i, n=struc->size, len, ai, check;
	unsigned long long j, n_vars=0;

	key = tr_get_tckey(tr, tcvc, distance, tup1, rel1, tup2, rel2, interp);
	/* look for this node in hash */
	hnode = hash_lookup(tr->hash->id_hash, key);

	if (hnode!=NULL)
	{
		ret = (struct dagnode *)hnode_get(hnode);
		tr_debug2("trd: hit %s in hash, returning\n", key);
		tr->hits++;
		free(key);
		return ret;
	}

	check = tr_check_equal_tup(struc,interp,tup1,tup2,rel1,rel2,tup_arity,
				   sorts);
	if (check==1)
	{
		free(key);
		return tr_get_true(tr);
	}
	else if (check == 0 && distance == 0) /* shouldn't happen, but safe */
	{
		free(key);
		return tr_get_false(tr);
	}
	else if (check == 2 && distance == 0)
	{
		free(key);
		return tr_make_equal_check(tr,struc,interp,rel1,rel2,tup_arity,
					   sorts);
	}
	if (distance == 1)
	{
		tmp = make_tr_tc1dagnode(struc,form,tr,tcvc,interp,tup1,rel1,
					 tup2,rel2);
		if (check == 2)
		{
			eq = tr_make_equal_check(tr,struc,interp,rel1,rel2,
						 tup_arity,sorts);
			ret = tr_get_dagnode(tr, TR_OR, tmp, eq, 0, 0);
			hash_alloc_insert(tr->hash->id_hash, key, ret);
			tr->keys = add_list(tr->keys, key);
			return ret;
		}
		hash_alloc_insert(tr->hash->id_hash, key, tmp);
		tr->keys = add_list(tr->keys, key);
		return tmp;
	}

	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
			k++;
		else
			n_vars += de_pow(n, sorts[i]);
	}

	ret = calloc(1, sizeof(struct dagnode));
	tmp = calloc(1, sizeof(struct dagnode));

	ret->type = TR_EXISTS;
	ret->num = tr->hash->next_id++;
	ret->child = malloc(sizeof(struct dagnode *));
	ret->child[0] = tmp;
	ret->n_child = 1;
	ret->n_vars = n_vars;
	ret->vars = malloc(sizeof(char *)*n_vars);

	m_vars = ret->vars;
	m_rels = malloc(sizeof(char *) * tup_arity);
	one_rels = malloc(sizeof(char *) * tup_arity);
	two_rels = malloc(sizeof(char *) * tup_arity);

	j=0;
	for (i=0; i<tup_arity; i++)
	{
		ai = sorts[i];
		if (ai==-1)
			continue;
		len = snprintf(NULL, 0, "_T%d_%lu", tcvc->num, tcvc->counter);
		m_rels[i] = malloc(sizeof(char)*(len+1));
		sprintf(m_rels[i], "_T%d_%lu", tcvc->num, tcvc->counter++);
		interp->rel_symbols = tr_insert_rel(interp->rel_symbols, 
						    m_rels[i], sorts[i]);
		tuple = NULL;
		while ((tuple = next_tuple(tuple, ai, n)))
			m_vars[j++] = tr_makevarname(tr,m_rels[i],0,tuple,ai);
	}

	tmp->type = TR_FORALL;
	tmp->num = tr->hash->next_id++;
	tmp->child = malloc(sizeof(struct dagnode *));
	tmp->n_child = 1;
	tmp->n_vars = 2 * n_vars;
	tmp->vars = malloc(sizeof(char *) * 2*n_vars);
	t_vars = tmp->vars;

	j=0;
	for (i=0; i<tup_arity; i++)
	{
		ai = sorts[i];
		if (ai == -1)
			continue;
		len = snprintf(NULL, 0, "_T%d_%lu", tcvc->num, tcvc->counter);
		one_rels[i] = malloc(sizeof(char) * (len+1));
		sprintf(one_rels[i], "_T%d_%lu", tcvc->num, tcvc->counter++);
		interp->rel_symbols = tr_insert_rel(interp->rel_symbols,
						    one_rels[i], sorts[i]);
		tuple = NULL;
		while ((tuple = next_tuple(tuple, ai, n)))
			t_vars[j++] = tr_makevarname(tr,one_rels[i],0,tuple,ai);
		len = snprintf(NULL, 0, "_T%d_%lu", tcvc->num, tcvc->counter);
		two_rels[i] = malloc(sizeof(char) * (len+1));
		sprintf(two_rels[i], "_T%d_%lu", tcvc->num, tcvc->counter++);
		interp->rel_symbols = tr_insert_rel(interp->rel_symbols,
						    two_rels[i], sorts[i]);
		tuple = NULL;
		while ((tuple = next_tuple(tuple, ai, n)))
			t_vars[j++] = tr_makevarname(tr,two_rels[i],0,tuple,ai);
	}

	psi = new_get_tr_tcdagnode(struc, form, tr, interp, tcvc, distance>>1,
				   tup1, one_rels, tup2, two_rels, index);
	tmp1 = tr_make_equal_check(tr,struc,interp,rel1,one_rels,
				   tup_arity,sorts);
	tmp2 = tr_make_equal_check(tr,struc,interp,m_rels,two_rels,
				   tup_arity,sorts);
	tmp1 = tr_get_dagnode(tr, TR_AND, tmp1, tmp2, 0, 0);

	tmp2 = tr_make_equal_check(tr,struc,interp,m_rels,one_rels,
				   tup_arity,sorts);
	tmp3 = tr_make_equal_check(tr,struc,interp,rel2,two_rels,
				   tup_arity,sorts);
	tmp2 = tr_get_dagnode(tr, TR_AND, tmp2, tmp3, 0, 0);

	condition = tr_get_dagnode(tr, TR_OR, tmp1, tmp2, 0, 0);
	/* condition -> psi === not condition OR psi */
	body = tr_get_dagnode(tr, TR_OR, condition, psi, 1, 0);
	tmp->child[0] = body;

	/* remove things added to interpretation */
	for (i=0; i<tup_arity; i++)
	{
		if (sorts[i]==-1)
			continue;
		rel = interp->rel_symbols;
		interp->rel_symbols = rel->next;
		free(rel);
		rel = interp->rel_symbols;
		interp->rel_symbols = rel->next;
		free(rel);
		rel = interp->rel_symbols;
		interp->rel_symbols = rel->next;
		free(rel);
		free(one_rels[i]);
		free(m_rels[i]);
		free(two_rels[i]);
	}

	/* free things */
	free(one_rels);
	free(two_rels);
	free(m_rels);

	/* add to hash */
	hash_alloc_insert(tr->hash->id_hash, key, ret);
	tr->keys = add_list(tr->keys, key);

	return ret;
}

struct dagnode *make_tr_tc(struct structure *struc, struct node *form,
		 	   struct tr *tr, struct interp *interp)
{
	struct node *tcargs=form->l->l, *relargs=form->r, *tmp;
	struct dagnode *ret=NULL;
	int *tup1, *tup2;
	char **rel1=NULL, **rel2=NULL;
	int *sorts;
	int *index=NULL;
	int tup_arity, size;
	int c, k, check1;
	unsigned long num_args, i, r;
	unsigned long long distance, d;
	struct tcvc *tcvc=NULL;
	int j, n_tcvc=tr->n_tcvc;

#if 0
	if (check_tc1(form)==0) /* TC1 */
		return old_make_tr_tc(struc, form, tr, interp);
#endif
	size=struc->size;

	for (tup_arity=0, tmp=tcargs; tmp; tup_arity++)
		tmp=tmp->r;
	for (i=0, tmp=relargs; tmp; i++)
		tmp=tmp->r;
	num_args = tup_arity<<1; /* parser does it with qnode */
	if (i!=num_args)
	{
		printf("tr31: TC of arity %lu given %lu arguments\n",num_args,i);
		tr->abort=1;
		return NULL;
	}

	for (j=0; j<n_tcvc; j++)
		if (form == tr->tcvc[j].tcform)
		{
			tcvc = tr->tcvc+j;
			break;
		}
	if (tcvc==NULL)
	{
		printf("tr32: TC formula not in tcvc, good luck\n");
		tr->abort=1;
		return NULL;
	}

	sorts = get_sorts(form);
	k=0;
	for (j=0; j<tup_arity; j++)
		if (sorts[j]>0)
			k++;

	if (k!=tup_arity) /* not pure TC2 */
	{
		free(sorts);
		if (check_tc1(form)==0)  /* TC1 */
			return old_make_tr_tc(struc, form, tr, interp);
		if (tr->warn == 0)
		{  /* mixed */
			tr->warn = 1;
			printf("tr33: warning: using exponential encoding for this TC\n");
		}
		return old_make_tr_tc(struc, form, tr, interp);
	}

	/* here we only handle TC2 (over only relations),
	 * mixed TC1/TC2 is preprocessed earlier.
	 * this could be redone for the general case
	 */
	for (j=0; j<tr->n_sovc; j++)
		tr->sovc[j].next++;

	tup1 = malloc(sizeof(int)*tup_arity);
	tup2 = malloc(sizeof(int)*tup_arity);
	rel1 = malloc(sizeof(char *)*tup_arity);
	rel2 = malloc(sizeof(char *)*tup_arity);
	if (k>0)
		index = malloc(sizeof(int)*k);

	c=0;
	tmp = relargs;
	for (j=0; j<tup_arity; j++,tmp=tmp->r)
	{
		if (sorts[j]==-1)
		{
			tup1[j] = teval(tmp->l, interp, struc);
			rel1[j] = NULL;
		}
		else
		{
			rel1[j] = tmp->l->data;
			index[c++] = j;
		}
	}
	for (j=0; j<tup_arity; j++,tmp=tmp->r)
	{
		if (sorts[j]==-1)
		{
			tup2[j] = teval(tmp->l, interp, struc);
			rel2[j] = NULL;
		}
		else
			rel2[j] = tmp->l->data;
	}

	/* maximum distance in the implicit graph is
	 *  D = n^r \Pi_{i=1}^k 2^{n^{a_i}} -1
	 * if there are r first-order variables, and k predicate variables
	 * per tuple.
	 *
	 * TC halves distance at each step, so smallest power of 2
	 * larger than that is convenient.
	 *
	 * log2 uses doubles, but size (n) will be small enough to
	 * be represented exactly.
	 */

	d=1, r=0;

	for (j=0; j<tup_arity; j++)
	{
		if (sorts[j] == -1)
			r++;
		else
			d *= de_pow(2,de_pow(size, sorts[j]));
	}
	d *= de_pow(size, r);


	for (distance=1; distance<d-1; distance = distance<<1)
		; /* empty loop, semicolon on new line to silence warning */

	/* do we need our formula? */
	check1 = tr_check_equal_tup(struc, interp, tup1, tup2, rel1, rel2, 
				   tup_arity, sorts);
	if (check1==1)
		ret = tr_get_true(tr);
	
	ret = new_get_tr_tcdagnode(struc, form, tr, interp, tcvc, distance,
				   tup1, rel1, tup2, rel2, index);
	free(tup1);
	free(tup2);
	free(sorts);
	free(rel1);
	free(rel2);
	free(index);
	return ret;
}

struct dagnode *to_qcir_rec(struct structure *struc, struct node *form,
			    struct tr *tr, struct interp *interp)
{
	struct dagnode *left=NULL, *right=NULL, *tmp1=NULL, *tmp2=NULL;
	int ileft, iright;

	if (tr->abort)
		return NULL;

	switch (form->label)
	{
		case TRUE:
			return tr_get_true(tr);
		case FALSE:
			return tr_get_false(tr);

		/* Boolean connectives */
		case NOT:
			tmp1 = to_qcir_rec(struc, form->l, tr, interp);
			return tr_get_dagnode(tr, TR_NOT, tmp1, NULL, 0, 0);
		case AND:
			left = to_qcir_rec(struc, form->l, tr, interp);
			right = to_qcir_rec(struc, form->r, tr, interp);
			return tr_get_dagnode(tr, TR_AND, left, right, 0, 0);
		case OR:
			left = to_qcir_rec(struc, form->l, tr, interp);
			right = to_qcir_rec(struc, form->r, tr, interp);
			return tr_get_dagnode(tr, TR_OR, left, right, 0, 0);
		case IMPLIES: /* a->b is (-a)|b */
			left = to_qcir_rec(struc, form->l, tr, interp);
			right = to_qcir_rec(struc, form->r, tr, interp);
			return tr_get_dagnode(tr, TR_OR, left, right, 1, 0);
		case IFF:  /* (a<->b) is (a&b)|(-a&-b) */
			left = to_qcir_rec(struc, form->l, tr, interp);
			right = to_qcir_rec(struc, form->r, tr, interp);
			tmp1 = tr_get_dagnode(tr, TR_AND, left, right, 0, 0);
			tmp2 = tr_get_dagnode(tr, TR_AND, left, right, 1, 1);
			return tr_get_dagnode(tr, TR_OR, tmp1, tmp2, 0, 0);
		case XOR:
			left = to_qcir_rec(struc, form->l, tr, interp);
			right = to_qcir_rec(struc, form->r, tr, interp);
			return tr_get_dagnode(tr, TR_XOR, left, right, 0, 0);

		/* quantifiers */
		case EXISTS:
			return make_tr_exists(struc, form, tr, interp);
		case FORALL:
			return make_tr_forall(struc, form, tr, interp);

		/* equals, nequals, etc */
		case LT:
		case LTE:
		case NEQUALS:
		case EQUALS:
			/* evaluate left and righthand sides, check values,
			 * output TRUE or FALSE
			 */
			ileft = teval(form->l, interp, NULL);
			iright = teval(form->r, interp, NULL);
			if (  (form->label==EQUALS && ileft==iright) ||
			      (form->label==NEQUALS&& ileft!=iright) ||
			      (form->label==LT     && ileft <iright) ||
			      (form->label==LTE    && ileft<=iright) )
			    return tr_get_true(tr);
			else /* return FALSE */
				return tr_get_false(tr);

		/* trickier ones */
		case PRED:
			return make_tr_pred(struc, form, tr, interp);
		case SOE:
			return make_tr_soe(struc, form, tr, interp);
		case TC:
			return make_tr_tc(struc, form, tr, interp);

		default:
			printf("tr20: unhandled node type %d\n", form->label);
			return NULL;
	}

	return NULL; /* unreachable */

}

void qcir_to_file_rec(struct dagnode *qcir, FILE *f)
{
	char *ops[] = {"unknown(", "exists(", "forall(", "and(", "or(", "xor(", "ite(", "and(-"};
	int i;
	unsigned int j; /* not really needed, but avoids warning on macos */

	if (qcir->printed) /* already did this node and its children */
		return;

	/* don't print child gate= var if it's a variable, just print the var
	 * directly below (remove one level of useless gate variables)
	 */
	for (i=0; i<qcir->n_child; i++)
		if (! (qcir->type >= TR_AND && qcir->type <= TR_NOT && 
		       qcir->child[i]->type==TR_VAR) )
				qcir_to_file_rec(qcir->child[i], f);

	qcir->printed = 1;	
	fprintf(f, "%d = ", qcir->num);

	if (qcir->type == TR_VAR)
	{
		fprintf(f, "and(%c%s)\n",
		  qcir->neg!=NULL && qcir->neg[0]? '-' : ' ',
		  qcir->vars[0]);
		return;
	}
	else if (qcir->type==TR_EXISTS || qcir->type==TR_FORALL)
	{
		fprintf(f, "%s", ops[qcir->type]);
		for (j=0; j<qcir->n_vars; j++)
			fprintf(f, "%c%s", (j!=0?',':' '), qcir->vars[j]);
		fprintf(f, "; %d)\n", qcir->child[0]->num);/* guaranteed one child */
		return;
	}
	else if (qcir->type>=TR_AND && qcir->type<TR_NOT)
	{
		fprintf(f, "%s", ops[qcir->type]);
		for (i=0; i<qcir->n_child; i++)
		{
			if (qcir->child[i]->type != TR_VAR)
				fprintf(f, "%c%c%d", (i!=0?',':' '),
				      (qcir->neg!=NULL && qcir->neg[i])?'-':' ',
					qcir->child[i]->num);
			else /* child is a variable, just print it here */
				fprintf(f, "%c%c%s",(i!=0?',':' '),
				   ((qcir->neg!=NULL&&qcir->neg[i]) ^
				    (qcir->child[i]->neg!=NULL && 
				     qcir->child[i]->neg[0]))?
				   '-':' ', qcir->child[i]->vars[0]);
		}
		fprintf(f, ")\n");
		return;
	}
	/* now we may produce TR_NOT with negated children */
	else if (qcir->type == TR_NOT)
	{	/* and(- is not, so use and( if the child is negated */
		if (qcir->child[0]->type != TR_VAR)
			fprintf(f, "and(%c%d)\n",
				(qcir->neg!=NULL&&qcir->neg[0])?' ':'-',
				qcir->child[0]->num);
		else /* child is a variable, just print it here */
		{
			fprintf(f, "and(%c%s)\n",
				((qcir->neg!=NULL&&qcir->neg[0]) ^
				 (qcir->child[0]->neg!=NULL &&
				  qcir->child[0]->neg[0]))?
				' ':'-', qcir->child[0]->vars[0]);
		}
		return;
	}
	else
	{
		printf("tr21: unknown dagnode type %d\n", qcir->num);
		return;
	}
}

/* output this qcir instance to f, QCIR-14 format */
/* if n_v is non-zero print it as the number of variables (cleansed style) */
void qcir_to_file(struct dagnode *qcir, FILE *f, unsigned long n_v, int command_type)
{
	struct dagnode *form = qcir;
	char *quant[] = {"", "exists(", "forall("};
	unsigned long i;

	if (n_v==0)
		fprintf(f, "#QCIR-G14\n");
	else
		fprintf(f, "#QCIR-G14 %lu\n", n_v);

	if (command_type == PQCIR) /* in prenex QCIR, print quantifiers here */
	{
		for (form=qcir; form->type==TR_EXISTS || form->type==TR_FORALL;
		     form = form->child[0])
		{
			/* print quantifier */
			fprintf(f,"%s",quant[form->type]);
			for (i=0; i<form->n_vars; i++)
				fprintf(f,"%c%s",(i==0?' ':','),form->vars[i]);
			fprintf(f,")\n");
		}
	}
	
	
	fprintf(f, "output(%d)\n", form->num);
	qcir_to_file_rec(form, f);

	return;
}

void free_renames(struct relrename *renames)
{
	struct relrename *tmp=renames, *next;

	while (tmp)
	{
		next = tmp->next;
		/*free(tmp->oldname);*/
		free(tmp->newname);
		free(tmp);
		tmp=next;
	}
	return;
}

void free_sovc(struct tr *tr)
{
	int i;

	for (i=0; i<tr->n_sovc; i++)
		free(tr->sovc[i].varname);
	free(tr->sovc);
	tr->sovc=NULL;
	tr->n_sovc=0;
	return;
}

struct list *make_dag_list(struct dagnode *dag, struct list *list)
{
	struct list *ret=list;
	int i;
	if (dag->printed == 2)
		return ret;

	for (i=0; i<dag->n_child; i++)
		ret = make_dag_list(dag->child[i], ret);

	dag->printed = 2;
	ret = add_list(ret, dag);
	return ret;
}

/* free the memory in d, not the subtree rooted there */
void free_dagnode(struct dagnode *d)
{
	unsigned long i;
	if (d==NULL)
		return;
	free(d->child);
	free(d->neg);
	for (i=0; i<d->n_vars; i++)
		free(d->vars[i]);
	free(d->vars);
	free(d->parents);
	free(d);
	return;
}

/* free all the memory used in the dag rooted here
 */
void dag_cleanup(struct dagnode *dag)
{
	struct list *dags = make_dag_list(dag, NULL);
	struct list *tmp, *next;
	struct dagnode *d;
	unsigned int i;

	for (tmp=dags; tmp; tmp=next)
	{
		next=tmp->next;
		d=tmp->data;
		free(d->child);
		free(d->neg);
		for (i=0; i<d->n_vars; i++)
			free(d->vars[i]);
		free(d->vars);
		free(d->parents);
		free(d);
		free(tmp);
	}
	return;
}


void dag_setrefs(struct dagnode *dag)
{
	int i;
	dag->refs++;
	if (dag->refs>1) /* already been here and set children */
		return;
	for (i=0; i<dag->n_child; i++)
		dag_setrefs(dag->child[i]);
}

/* dag is no longer needed (ref count 0)
 *  put it in the list of refcount zero dagnodes in tr, to
 *  possibly free later if refs doesn't increase.
 * if c is not 0, also look at descendants - if this dag has the
 * only references, also put them in the list.
 */
void clean_dnode(struct tr *tr, struct dagnode *dag, int c)
{
	struct list *tmp;
	int i, flag=0;
	assert(dag->refs==0);

	for (tmp=tr->unused; tmp; tmp=tmp->next)
		if (tmp->data == dag)
			flag=1;

	if (!flag)
		tr->unused = add_list(tr->unused, dag);
	if (c==0)
		return;

	for (i=0; i<dag->n_child; i++)
	{
		dag->child[i]->refs--;
		if (dag->child[i]->refs==0)
			clean_dnode(tr, dag->child[i], c);
	}
	
	return; 
}

/* if we have a child of type TR_NOT, try to go to its child using dag->neg
 * instead, changing refs, etc as needed
 */
struct dagnode *dag_simplify_rec_not(struct tr *tr, struct dagnode *dag)
{
	struct dagnode *child;
	int i;
	if (dag->simplified<2)
		for (i=0; i<dag->n_child; i++)
			dag->child[i] = dag_simplify_rec_not(tr, dag->child[i]);
	dag->simplified = 2;

	for (i=0; i<dag->n_child; i++)
	{
		child = dag->child[i];
		if (child->type != TR_NOT)
			continue;
		if (child->neg==NULL || child->neg[0]==0)
		{ /* if NOT NOT don't need the negation at all */
			if (dag->neg==NULL)
				dag->neg = calloc(dag->n_child, sizeof(int));
			if (dag->neg[i])
				dag->neg[i] = 0;
			else
				dag->neg[i] = 1;
		}
		dag->child[i] = child->child[0];
		child->refs--;
		if (child->refs == 0) /* removing entirely */
			clean_dnode(tr, child, 0); /* we re-ref only child */
	}
	return dag;
}

struct dagnode *dag_simplify_rec(struct tr *tr, struct dagnode *dag)
{
	struct dagnode **ch, *tru, *fal, *ret;
	int *neg, i, j, tcount=0, fcount=0, n_ch, left, right, flag;

	if (!dag->simplified)
		for (i=0; i<dag->n_child; i++)
			dag->child[i] = dag_simplify_rec(tr, dag->child[i]);
	dag->simplified=1;

	if (dag->type == TR_VAR) /* can't simplify single variable */
		return dag;

	tru = tr_get_true(tr);
	fal = tr_get_false(tr);

	if (dag == tru || dag == fal) /* can't simplify "true" or "false" */
		return dag;

	for (i=0; i<dag->n_child; i++)
	{
		if ( (dag->child[i] == tru && (!dag->neg || !dag->neg[i])) ||
		     (dag->child[i] == fal && dag->neg && dag->neg[i]) )
			tcount++;
		if ( (dag->child[i] == fal && (!dag->neg || !dag->neg[i])) ||
		     (dag->child[i] == tru && dag->neg && dag->neg[i]) )
			fcount++;
	}

	if (tcount==0 && fcount==0)
		return dag;

	/* otherwise we have at least some children that are true/false */
	switch (dag->type)
	{
		/* quantifiers only have one child.
		 * if it's true or false, no need for the quantifier at all
		 */
		case TR_EXISTS:
		case TR_FORALL:
			assert(dag->n_child==1);
			if (tcount==1) /* ret the child, ref count unchanged*/
				ret = tr_get_true(tr);
			else
				ret = tr_get_false(tr);
			dag->refs--;
			ret->refs++;
			if (dag->refs == 0) /* removing entirely */
				clean_dnode(tr, dag,1);
			return ret;
		case TR_NOT:
			assert(dag->n_child==1);
			if ( tcount==1 )
				ret = tr_get_false(tr);
			else
				ret = tr_get_true(tr);
			dag->refs--;
			ret->refs++;
			if (dag->refs == 0)
				clean_dnode(tr, dag,1);
			return ret;
		case TR_AND:
			if (fcount>0) /* AND with a false child */
			{
				ret = tr_get_false(tr);
				ret->refs++;
				dag->refs--;
				if (dag->refs == 0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			/* otherwise tcount>0 */
			else if (tcount == dag->n_child) /* AND with all true */
			{
				ret = tr_get_true(tr);
				ret->refs++;
				dag->refs--;
				if (dag->refs == 0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			/* otherwise, tcount>0, remove true children */
			n_ch = dag->n_child - tcount;
			ch = malloc(sizeof(struct dagnode *)*n_ch);
			neg = NULL;
			if (dag->neg!=NULL)
				neg = malloc(sizeof(int)*n_ch);
			j=0;
			for (i=0; i<dag->n_child; i++)
			{
				if (dag->child[i] == tru || dag->child[i]==fal)
				{
					dag->child[i]->refs--;
					if (dag->child[i]->refs==0)
						clean_dnode(tr,dag->child[i],0);
				}
				else
				{
					if (dag->neg)
						neg[j]=dag->neg[i];
					ch[j++]=dag->child[i];
				}
			}
			free(dag->child);
			free(dag->neg);
			dag->child = ch;
			dag->n_child = n_ch;
			dag->neg = neg;
			return dag;

		case TR_OR:
			if (tcount>0) /* OR with a true child */
			{
				//return dag; /* TODO: memory leak if use below?*/
				ret = tr_get_true(tr);
				ret->refs++;
				dag->refs--;
				if (dag->refs == 0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			/* otherwise fcount>0 */
			else if (fcount == dag->n_child) /* OR with all false */
			{
				ret = tr_get_false(tr);
				ret->refs++;
				dag->refs--;
				if (dag->refs == 0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			/* otherwise fcount>0, remove false children */
			n_ch = dag->n_child - fcount;
			ch = malloc(sizeof(struct dagnode *)*n_ch);
			neg = NULL;
			if (dag->neg != NULL)
				neg = malloc(sizeof(int)*n_ch);
			j=0;
			for (i=0; i<dag->n_child; i++)
			{
				if (dag->child[i] == fal || dag->child[i]==tru)
				{
					dag->child[i]->refs--;
					if (dag->child[i]->refs==0)
						clean_dnode(tr,dag->child[i],0);
				}
				else
				{
					if (dag->neg)
						neg[j]=dag->neg[i];
					ch[j++]=dag->child[i];
				}
			}
			free(dag->child);
			free(dag->neg);
			dag->child = ch;
			dag->n_child = n_ch;
			dag->neg = neg;
			return dag;

		case TR_ITE: /* not used yet, so don't bother */
			return dag;

		case TR_XOR:
			assert(dag->n_child == 2);
			if (tcount + fcount == dag->n_child)
			{	/* either true or false */
				if ((dag->child[0]==tru && (dag->neg==NULL ||
							   !dag->neg[0])) ||
				    (dag->child[0]==fal&&dag->neg&&dag->neg[0]))
					left = 1;
				else
					left = 0;
				if ((dag->child[1]==tru && (dag->neg==NULL ||
							    !dag->neg[1])) ||
				    (dag->child[1]==fal&&dag->neg&&dag->neg[1]))
					right = 1;
				else
					right = 0;
				if (left ^ right)
					ret = tr_get_true(tr);
				else
					ret = tr_get_false(tr);
				ret->refs++;
				dag->refs--;
				if (dag->refs==0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			/* otherwise, we have one constant child and one dnode*/
			/* return dnode or not dnode as needed */
			if (dag->child[0]==tru || dag->child[0]==fal)
				i=1,j=0;
			else
				i=0,j=1;
			/*i is the non-constant child, j is the constant child*/
			flag = 0; /* is the XOR over NOT child[i] ? */
			if (dag->neg && dag->neg[i])
				flag=1;
			if ((dag->child[j]==fal && (!dag->neg || !dag->neg[j])&&			     flag == 0) ||
			    (dag->child[j]==tru && dag->neg && dag->neg[j] &&
			     flag == 0) ||
			    (dag->child[j]==fal && dag->neg && dag->neg[j] &&
			     flag == 1) ||
			    (dag->child[j]==tru && (!dag->neg || !dag->neg[j])&&
			     flag == 1))
			{
				ret = dag->child[i];
				ret->refs++;
				dag->refs--;
				if (dag->refs==0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
			else
			{
				ret = calloc(1, sizeof(struct dagnode));
				ret->type = TR_NOT;
				ret->child = malloc(sizeof(struct dagnode *));
				ret->child[0] = dag->child[i];
				ret->refs++;
				ret->n_child = 1;
				dag->child[i]->refs++;
				dag->refs--;
				if (dag->refs==0)
					clean_dnode(tr, dag, 1);
				return ret;
			}
		default:
			printf("tr36: unknown node (%d) in simplify\n",
				dag->type);
			return dag;
	}
	return dag; /* unreachable */
}

/* simplify the given formula. for now, just propagates true/false.
 */
struct dagnode *dag_simplify(struct tr *tr, struct dagnode *dag)
{
	struct dagnode *ret, *d;
	struct list *tmp, *next;
	unsigned long i;

	dag_setrefs(dag);
	ret = dag_simplify_rec(tr, dag);
	ret = dag_simplify_rec_not(tr, ret); /* try to remove TR_NOT using neg */

	for (tmp=tr->unused; tmp; tmp=next)
	{
		next = tmp->next;
		d = tmp->data;
		if (d->refs>0
			|| d==&dagn_true || d==&dagn_false
			/* don't free the hacked in static dagn_* */
		   )
		{ /* it was unused, but we referenced it again, so back in dag*/
			free(tmp); /* still need to free the list entry */
			continue;
		}
		free(d->child);
		free(d->neg);
		for (i=0; i<d->n_vars; i++)
			free(d->vars[i]);
		free(d->vars);
		free(d);
		free(tmp);
	}
	return ret;
}

