/*
Copyright (c) 2022, Charles Jordan <skip@res.otaru-uc.ac.jp>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/
/* convert.c
 * Skip Jordan
 *
 * Given a dagnode for non-cleansed, non-prenexed QCIR, this file supports
 * the following:
 *  1) Cleansing
 *  2) Prenexing
 *  3) Conversion to prenex CNF
 *  4) Printing prenex CNF as QDIMACS
 *
 * The above need to be done in order: prenexing only works after cleansing,
 * converting to PCNF requires being cleansed and prenexed.
 *
 * Currently qdimacs conversion doesn't handle 0-ary AND/OR (QCIR true/false)
 * constants. So one should use QCIRHACK; this is handled in trans.c.
 *
 * chj	10/13/22	created
 * chj	10/25/22	cleansing, prenexing, qdimacsing
 *
 * last output number: c07
 */

#include "parse.h"
#include "deqbf.h"
#include "hash.h"
#include "protos.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

/* replace any xor or ite gates by equivalent */
/* part of cleansing */
struct dagnode *remove_xor_ite(struct tr *tr, struct dagnode *qcir)
{
	struct dagnode *left, *right;
	int i, lneg, rneg, cneg;
	switch (qcir->type)
	{
		case TR_EXISTS:
		case TR_FORALL:
			qcir->child[0] = remove_xor_ite(tr, qcir->child[0]);
			return qcir;

		case TR_AND:
		case TR_OR:
			for (i=0; i<qcir->n_child; i++)
				qcir->child[i] = remove_xor_ite(tr,
								qcir->child[i]);
			return qcir;

		case TR_XOR:
			qcir->child[0] = remove_xor_ite(tr, qcir->child[0]);
			qcir->child[1] = remove_xor_ite(tr, qcir->child[1]);
			lneg = 0;
			rneg = 0;
			if (qcir->neg!=NULL && qcir->neg[0])
				lneg = 1;
			if (qcir->neg!=NULL && qcir->neg[1])
				rneg = 1;

			left = tr_get_dagnode(tr, TR_AND,
					      qcir->child[0], qcir->child[1],
					      (lneg?1:0), (rneg?0:1));
			right = tr_get_dagnode(tr, TR_AND,
					       qcir->child[0], qcir->child[1],
					      (lneg?0:1), (rneg?1:0));
			qcir->type = TR_OR;
			qcir->child[0] = left;
			qcir->child[1] = right;
			if (qcir->neg)
			{
				qcir->neg[0] = 0;
				qcir->neg[1] = 0;
			}
			return qcir;

		case TR_ITE: /* not used currently, but supported here */
			qcir->child[0] = remove_xor_ite(tr, qcir->child[0]);
			qcir->child[1] = remove_xor_ite(tr, qcir->child[1]);
			qcir->child[2] = remove_xor_ite(tr, qcir->child[2]);
			lneg = 0;
			cneg = 0;
			rneg = 0;
			if (qcir->neg!=NULL && qcir->neg[0])
				lneg = 1;
			if (qcir->neg!=NULL && qcir->neg[1])
				cneg = 1;
			if (qcir->neg!=NULL && qcir->neg[2])
				rneg = 1;

			left = tr_get_dagnode(tr, TR_AND,
					      qcir->child[0], qcir->child[1],
					      (lneg?1:0), (cneg?1:0));
			right = tr_get_dagnode(tr, TR_AND,
					      qcir->child[0], qcir->child[2],
					      (lneg?0:1), (rneg?1:0));
			qcir->type = TR_OR;
			qcir->child[0] = left;
			qcir->child[1] = right;
			qcir->n_child = 2;
			if (qcir->neg)
			{
				qcir->neg[0] = 0;
				qcir->neg[1] = 0;
				qcir->neg[2] = 0;
			}
			return qcir;

		case TR_NOT:
			qcir->child[0] = remove_xor_ite(tr, qcir->child[0]);
			return qcir;

		case TR_VAR:
			return qcir;

		default:
			printf("c01: unknown dagnode label %d\n", qcir->type);
			return qcir;		
	}
	return qcir; /* unreachable */
}

/* fill out the parents and n_parents fields in the subdag rooted here
 */
void set_qcir_parents(struct dagnode *qcir)
{
	int i, j, hit;
	struct dagnode **tmp, *child;
	if (qcir==NULL)
		return;

	for (i=0; i<qcir->n_child; i++)
	{
		child = qcir->child[i];
		if (child->n_parents == 0) /* haven't recursed yet*/
			set_qcir_parents(child);
		hit = 0;
		for (j=0; j<child->n_parents; j++)
			if (child->parents[j] == qcir)
			{
				hit=1;
				break;
			}
		if (hit == 1) /* already added as a parent */
			continue;

		/* add qcir as next parent */
		tmp = realloc(child->parents, 
			      sizeof(struct dagnode*) * (child->n_parents+1));
		if (tmp==NULL)
		{
			printf("c02: error, unable to allocate memory.\n");
			return;
		}
		child->parents = tmp;
		child->parents[child->n_parents] = qcir;
		child->n_parents++;
	}
}

/* we want to return a duplicate copy of qcir, that respects the cleansed
 * qcir requirement that nodes with quantified subformulas are used only
 * once.
 * this means we could avoid duplicating some subformulas.
 * for now, we duplicate more than required.
 * if hard==0 we need to duplicate qcir, but we re-use
 * any children with qsf==-1. once hitting a quantifier, we use hard=1
 * duplication and duplicate everything.
 * TODO: track dependencies and be more clever
 */
struct dagnode *qcir_duplicate(struct tr *tr, struct dagnode *qcir, int hard)
{
	struct dagnode *ret, *child, **tmp;
	int i, nc, nv;

	/* if quantifier, we duplicate everything below */
	if (hard == 0 && (qcir->type == TR_EXISTS || qcir->type == TR_FORALL))
		return qcir_duplicate(tr, qcir, 1);

	ret = calloc(1, sizeof(struct dagnode));
	ret->type = qcir->type;
	ret->num = tr->hash->next_id++;
	nc = qcir->n_child;
	ret->n_child = nc;
	if (nc>0)
		ret->child = malloc(sizeof(struct dagnode *)*nc);
	nv = qcir->n_vars;
	ret->n_vars = nv;
	if (nv>0)
		ret->vars = malloc(sizeof(char *) * nv);
	if (qcir->neg)
		ret->neg = calloc(nv+nc, sizeof(int));

	/* handle children */
	for (i=0; i<nc; i++)
	{
		child = qcir->child[i];
		if (hard==0 && child->qsf==-1)
			ret->child[i] = child; /* re-use */
		else
			ret->child[i] = qcir_duplicate(tr, child, hard);
		child = ret->child[i];
		tmp = realloc(child->parents, sizeof(struct dagnode*)*
			      (1+child->n_parents));
		if (tmp==NULL)
		{
			printf("c03: error no memory\n");
			return NULL;
		}
		child->parents = tmp;
		child->parents[child->n_parents++] = ret;
	}

	/* handle vars */
	for (i=0; i<nv; i++)
		ret->vars[i] = dupstr(qcir->vars[i]);

	if (qcir->neg)
		for (i=0; i<nv+nc; i++)
			ret->neg[i] = qcir->neg[i];

	ret->qsf = qcir->qsf;
	ret->defanged = qcir->defanged;

	return ret;
}

/* duplicate (triplicate, etc) any dagnode a such that
 *  1) a has n_parents>1
 *  2) a has a quantified subformula
 *
 * we must work bottom up: handle children before the current node
 */
void handle_reused_qsf(struct tr *tr, struct dagnode *qcir) 
{
	struct dagnode *parent, *clone;
	int i, j, hit;

	if (qcir->qsf != 1) /* no quantified subformulas in here */
		return;

	for (i=0; i<qcir->n_child; i++)
		handle_reused_qsf(tr, qcir->child[i]);

	/* now we've handled the children */

	if (qcir->n_parents == 1) /* only one parent, okay as is */
	{
		qcir->defanged = 1;
		return; 
	}

	/* we need to make one copy of qcir for each parent.
	 * this involves duplicating any nodes in the subdag
	 */
	for (i=1; i<qcir->n_parents; i++) /* parents[0] gets original copy */
	{
		parent = qcir->parents[i]; /* others get a clone */
		hit = 0;
		for (j=0; j<parent->n_child; j++)
			if (parent->child[j] == qcir)
			{
				clone = qcir_duplicate(tr, qcir, 0);
				parent->child[j] = clone;
				clone->parents=malloc(sizeof(struct dagnode *));
				clone->parents[0] = parent;
				clone->n_parents = 1;
				hit = 1;
				break;
			}
		assert(hit == 1);
	}
	qcir->defanged = 1;
}

/* set qsf=1 in ancestors of this node.
 * if we hit an ancestor already with qsf=1, can stop recursing (since
 * already did ancestors)
 */
void qsf_parents(struct dagnode *qcir)
{
	int i;

	if (qcir->qsf == 1)
		return;

	for (i=0; i<qcir->n_parents; i++)
		qsf_parents(qcir->parents[i]);

	qcir->qsf = 1;
	return;
}

/* set_qsf()
 * sets qsf to 1 in any node that contains a quantified subformula
 * needs parents / n_parents set in all dagnodes
 */
void set_qsf(struct dagnode *qcir)
{
	int i;

	if (qcir==NULL)
		return;

	for (i=0; i<qcir->n_child; i++)
		if (qcir->child[i]->qsf == 0)
			set_qsf(qcir->child[i]);

	if ( (qcir->type == TR_EXISTS || qcir->type == TR_FORALL) &&
	     qcir->qsf == 0 )
	{
		qcir->qsf = 1;
		for (i=0; i<qcir->n_parents; i++)
			qsf_parents(qcir->parents[i]);
	}

	if (qcir->qsf == 0)
		qcir->qsf = -1;

	return;
}

void rename_distinct_rec(struct tr *tr, struct dagnode *qcir,
			 unsigned long *next, int *label)
{
	int *oldv, *inames;
	char **oldnames;
	int oldn, newn, len, nn, i;

	if (qcir==NULL || qcir->defanged==2 || qcir->defanged==4)
		return;

	/* first, we rename this node */
	if (qcir->type != TR_VAR)
	{
		oldn = qcir->num;
		if (label[oldn]!=0)
			newn = label[oldn];
		else
			newn = label[oldn] = (*next)++;
		qcir->num = newn;
	}
	else
		qcir->num = 0;	/* we don't print TR_VAR gate variables */
				/* instead just printing the variable directly*/
				/* see trans.c:qcir_to_file_rec() */

	/* next we handle children and special cases */
	switch (qcir->type)
	{
		case TR_EXISTS: /* most special case, this is okay? */
		case TR_FORALL:
			nn = qcir->n_vars;
			assert(nn>0);
			oldv = malloc(sizeof(int)*nn);
			inames = malloc(sizeof(int)*nn);
			oldnames = malloc(sizeof(char *) * nn);
			for (i=0; i<nn; i++)
			{ /* save previous renames */
				oldnames[i] = qcir->vars[i];
				inames[i] = atoi(oldnames[i]);
				oldv[i] = label[inames[i]];
				free(oldnames[i]);
			}
			for (i=0; i<nn; i++)
			{ /* make new renames */
				len = snprintf(NULL, 0, "%lu", *next);
				qcir->vars[i] = malloc(sizeof(char)*(len+1));
				sprintf(qcir->vars[i], "%lu", *next);
				label[inames[i]] = (*next)++;
			}
			for (i=0; i<qcir->n_child; i++) /* do renames */
				rename_distinct_rec(tr,qcir->child[i],next,label);
			for (i=0; i<nn; i++) /* done this subform, restore old*/
				label[inames[i]] = oldv[i];
			free(oldv);
			free(inames);
			free(oldnames);
			break;

		case TR_AND:
		case TR_OR:
		case TR_XOR: /* removed already, not needed */
		case TR_ITE: /* not used, removed if used, not needed */
		case TR_NOT:
			for (i=0; i<qcir->n_child; i++)
				rename_distinct_rec(tr,qcir->child[i],next,label);
			break;
		case TR_VAR:
			oldn = atoi(qcir->vars[0]);
			assert(label[oldn]!=0); /* no free variables */
			if (label[oldn]!=0)
				newn = label[oldn];
			else /* certainly shouldn't have free variables */
				newn = label[oldn] = (*next)++;
			free(qcir->vars[0]);
			len = snprintf(NULL, 0, "%d", newn);
			qcir->vars[0] = malloc(sizeof(char)*(len+1));
			sprintf(qcir->vars[0], "%d", newn);
			break;	
		default:
			printf("c04: unknown label %d\n", qcir->type);
			break;
	}
	qcir->defanged = 2;
	return;
}

/* we've done remove_xor_ite, set_qcir_parents, set_qsf, handle_reused_qsf().
 * 
 * next, we need to rename gates and variables and set *n_v such that:
 *  1) variables are quantified only once.    (ie not re-used)
 *  2) variables and gates are named 1...*n_v (ie no gaps)
 *
 * we've already done handle_reused_qsf() so no quantified subformulas are
 * reachable from multiple paths.
 */
void rename_distinct_vars(struct tr *tr,struct dagnode *qcir,unsigned long *n_v)
{
	int *label = calloc(tr->hash->next_id, sizeof(int)+1);
	/* we're renumbering i as label[i] */
	unsigned long next = 1; /* this is the next label to use */

	rename_distinct_rec(tr,qcir,&next,label);

	free(label);
	*n_v = next-1;

	return;
}

struct dagnode *qcir_cleanse(struct tr *tr, struct dagnode *qcir,
			     unsigned long *n_v)
{
	struct dagnode *ret;

	if (qcir==NULL)
		return NULL;

	ret = remove_xor_ite(tr, qcir);
	set_qcir_parents(ret);

	set_qsf(ret);
	handle_reused_qsf(tr, ret);

	*n_v = 0;
	
	rename_distinct_vars(tr, ret, n_v);

	return ret;
}

/* we need to move any quantified subformulas up, respecting
 * negations as needed
 * no guarantee on parents fields being usable after this.
 * no guarantee on variable names without holes after this - otherwise
 * cleansed formulas maintain their properties (ie rerun rename variables)
 */
struct dagnode *qcir_prenex_int(struct tr *tr, struct dagnode *qcir)
{
	struct dagnode *ret, *child, *ex, *un;
	int label;
	int i, n_e, n_u, cex, cun, qsf=-1;
	unsigned long j;

	if (qcir==NULL
/*#if 0 */
/* fixed 11/24/22 should be okay to not recheck things with qsf==1 again */
		|| qcir->qsf==-1
/*#endif*/
	   )
		return qcir; /* done or no quantifiers here */

	for (i=0; i<qcir->n_child; i++)
	{
		qcir->child[i] = qcir_prenex_int(tr, qcir->child[i]);
		if (qcir->child[i]->type == TR_EXISTS || qcir->child[i]->type==TR_FORALL)
			qsf=1;
	}
	if (qsf==-1) /* no quantified children */
		return qcir;

	/* if qcir is a NOT, handle it */
	if (qcir->type == TR_NOT)
	{
		label = qcir->child[0]->type;
		assert(label==TR_EXISTS || label==TR_FORALL); /* qsf==1 */
		if (qcir->neg && qcir->neg[0]) /* NOT NOT */
		{ /* we need to rename vars again later */
			ret = qcir->child[0];
			free_dagnode(qcir);
			return ret;
		}
		/* not \Q \psi is \Q' not \psi */
		child = qcir->child[0]; /* original quantifier */
		ret = qcir;
		ret->vars = child->vars;
		ret->n_vars = child->n_vars;
		/* both have n_child == 1 */
		if (child->type == TR_EXISTS)
			ret->type = TR_FORALL;
		else
			ret->type = TR_EXISTS;
		child->type = TR_NOT;
		child->n_vars = 0;
		child->vars = NULL;
		/* we moved NOT below a quantifier, could be no longer
		 * has quantified subformulas
		 */
		ret->child[0]->qsf = ret->child[0]->child[0]->qsf;
		/* move NOT down */
		ret->child[0] = qcir_prenex_int(tr, ret->child[0]);
		/* ret is now a quantifier so obviously qsf==1 */
		ret->qsf = 1;
		return ret;
	}

	/* handle any negated quantified children */
	for (i=0; i<qcir->n_child; i++)
	{
		child = qcir->child[i];
		label = child->type;
		if ( (label==TR_EXISTS || label==TR_FORALL) &&
		     (qcir->neg && qcir->neg[i]) )
		{
			qcir->neg[i] = 0;
			if (label==TR_EXISTS)
				child->type = TR_FORALL;
			else
				child->type = TR_EXISTS;
			/* negate body */
			if (child->neg && child->neg[0])
				child->neg[0] = 0;
			else if (child->neg && child->neg[0]==0)
				child->neg[0] = 1;
			else /* child->neg is NULL */
			{
				child->neg = malloc(sizeof(int)*1);
				child->neg[0] = 1;
			}
		}
	}

	n_e = 0;
	n_u = 0;
	/* now, we need to make new leading quantifiers.
	 * quantifiers don't overlap so any order is okay.
	 * we put existential first, then universal.
	 */
	for (i=0; i<qcir->n_child; i++)
	{
		child = qcir->child[i];
		if (child->type == TR_EXISTS)
			n_e += child->n_vars;
		else if (child->type == TR_FORALL)
			n_u += child->n_vars;
	}

	if (n_e==0 && n_u==0)	/* this node had a quantified subformula, */
	{			/* but we moved the quantifiers up and */
		qcir->qsf = -1;	/* now it has no quantified subformulas. */
		return qcir;	/* so it's prenexed as is */
	}

	if (qcir->type == TR_EXISTS || qcir->type == TR_FORALL)
	{ /* quantified root with non-negated prenex children is already prenex
	   * we could combine adjacent same quantifiers but not required in qcir	   */
		return qcir;
	}
	/* otherwise we have quantified children and we're not a quantifier */
	/* first, move quantified children above, moving the current formula
	 * (with any non-quantified subformulas) down.
	 * then we need to re-prenex the child of the new leading quantifier
	 * block.
	 */
	if (n_e>0)
	{
		ex = calloc(1, sizeof(struct dagnode));
		ex->type = TR_EXISTS;
		ex->num = tr->hash->next_id++;
		ex->n_vars = n_e;
		ex->vars = malloc(sizeof(char *)*n_e);
		ex->n_child = 1;
		ex->child = malloc(sizeof(struct dagnode *));
		ex->defanged = 4;
		ex->qsf = 1;
	}
	if (n_u>0)
	{
		un = calloc(1, sizeof(struct dagnode));
		un->type = TR_FORALL;
		un->num = tr->hash->next_id++;
		un->n_vars = n_u;
		un->vars = malloc(sizeof(char *)*n_u);
		un->n_child = 1;
		un->child = malloc(sizeof(struct dagnode *));
		un->defanged = 4;
		un->qsf = 1;
	}

	if (n_e>0)
		ret = ex;
	else
		ret = un;
	if (n_u>0)
		un->child[0] = qcir;
	else
		ex->child[0] = qcir;

	/* for each child, if quantified, move the vars to ex/un, move
	 * \phi to qcir->child[i], handle negated phi appropriately
	 */
	cex = 0;
	cun = 0;
	if (qcir->neg == NULL)
		qcir->neg = calloc(qcir->n_child, sizeof(int));
	for (i=0; i<qcir->n_child; i++)
	{
		child = qcir->child[i];
		if (child->type == TR_EXISTS)
			for (j=0; j<child->n_vars; j++)
				ex->vars[cex++] = child->vars[j];
		else if (child->type == TR_FORALL)
			for (j=0; j<child->n_vars; j++)
				un->vars[cun++] = child->vars[j];
		else
			continue;

		qcir->child[i] = child->child[0];
		if (child->neg && child->neg[0])
			qcir->neg[i] = 1; /* above we handled negated quantified
				           * children, so safe to negate here
					   * TODO: REALLY?*/
		free(child->vars);
		child->n_vars = 0;
		child->vars = NULL;
		free_dagnode(child); /* child was quantified, so only referred
				      * to once (here) because it was cleansed*/
	}

	/* since we moved things, we need to re-prenex \phi */
	if (n_u>0)
		un->child[0] = qcir_prenex_int(tr, un->child[0]);
	else
		ex->child[0] = qcir_prenex_int(tr, ex->child[0]);
	return ret;
}

struct dagnode *qcir_prenex(struct tr *tr, struct dagnode *qcir, 
			    unsigned long *n_v)
{
	struct dagnode *ret;
	ret = qcir_prenex_int(tr,qcir);
	rename_distinct_vars(tr, ret, n_v);
	return ret;
}

int dvar(struct dagnode *qcir)
{
	if (qcir->type != TR_VAR)
		return qcir->num;
	else
		return atoi(qcir->vars[0]);
}

/* make clauses for dagnode qcir, using variable var as the new variable
 * for this node.
 * requires qcir to be cleansed and prenexed, and qcir starts from the
 * body of the circuit, not leading quantifiers
 */
struct list *qcir_to_qdimacs_rec(struct dagnode *qcir,
				 struct list *oclauses, unsigned int *n,
				 unsigned int *m, unsigned int *vars,
				 unsigned int *uvars)
{
	struct list *ret;
	char *clauses;
	unsigned int len, i, nc;
	int negi, cd;
	if (qcir == NULL || qcir->defanged>=5)
		return oclauses;
	qcir->defanged = 5;
	switch (qcir->type)
	{
		case TR_EXISTS:
		case TR_FORALL:
			printf("c05: trying to convert non-prenex formula\n");
			return oclauses;
		case TR_AND:
		case TR_OR:
			uvars[qcir->num] = 1;
			nc = qcir->n_child;
			len = 0;
			len += snprintf(NULL, 0, "%d", qcir->num);
			len *= nc+1;
			for (i=0; i<nc; i++)
				len += 2 * snprintf(NULL, 0, " %d", 
						    dvar(qcir->child[i]));
			len += 2*(nc+1); /* " 0" at end of nc+1 clauses */
			len += nc+1; /* nc+1 negations */
			len += nc+1; /* nc+1 newlines */
			len += 2*nc; /* negations/extra spaces from neg */
			len += 1; /* closing \0 */
			if (qcir->type == TR_AND)
				len += nc; /* other negations */
			clauses = malloc(sizeof(char)*len);

			if (qcir->type == TR_OR)
				len = sprintf(clauses, "-%d", qcir->num);
			else
				len = sprintf(clauses, "%d", qcir->num);
			for (i=0; i<nc; i++)
			{
				negi = ( qcir->neg!=NULL && qcir->neg[i] );
				if (qcir->type == TR_OR)
					len += sprintf(clauses+len, " %c%d",
							negi?'-':' ',
							dvar(qcir->child[i]));
				else
					len += sprintf(clauses+len, " %c%d",
							negi?' ':'-',
							dvar(qcir->child[i]));
			}
			len += sprintf(clauses+len, " 0\n");
			for (i=0; i<nc; i++)
			{
				negi = ( qcir->neg!=NULL && qcir->neg[i] );
				if (qcir->type == TR_OR)
					len +=sprintf(clauses+len,"%d %c%d 0\n",
						qcir->num,negi?' ':'-',
						dvar(qcir->child[i]));
				else
					len+=sprintf(clauses+len,"-%d %c%d 0\n",
						qcir->num,negi?'-':' ',
						dvar(qcir->child[i]));
			}
			*m += (nc+1);
			ret = add_list(oclauses, clauses);
			vars[*n]=qcir->num;
			*n += 1;
			for (i=0; i<nc; i++)
			{
				if (qcir->child[i]->type != TR_VAR)
				   ret = qcir_to_qdimacs_rec(qcir->child[i],ret,
							     n, m, vars, uvars);
				else
					uvars[dvar(qcir->child[i])] = 1;
			}
			return ret;
		case TR_ITE:
		case TR_XOR:
			printf("c06: trying to convert non-cleansed formula to CNF\n");
			return oclauses;

		case TR_NOT:
			/* (g or x) and (not g or not x) */
			negi = (qcir->neg && qcir->neg[0]);
			vars[*n] = qcir->num;
			*n += 1;
			*m += 2;
			uvars[qcir->num] = 1;
			cd = dvar(qcir->child[0]);
			len = snprintf(NULL, 0, "%d  %d 0\n-%d  %d 0\n",
					qcir->num,cd,qcir->num,cd);
			clauses = malloc(sizeof(char)*(len+1));
			sprintf(clauses, "%d %c%d 0\n-%d %c%d 0\n",qcir->num,
				(negi?'-':' '),cd,qcir->num,(negi?' ':'-'),cd);
			ret = add_list(oclauses, clauses);
			if (qcir->child[0]->type != TR_VAR)
				ret = qcir_to_qdimacs_rec(qcir->child[0], ret,
							  n, m, vars, uvars);
			else
				uvars[cd] = 1;
			return ret;
		default:
			printf("c07: unknown type %d in CNF conversion\n",
				qcir->type);
			return oclauses;
	}
	return oclauses; /* unreachable */
}

/* add silly clause (var or not var) to oclauses
 * this is to avoid the case where we have a quantified but unused variable
 */
struct list *pointless_clauses(struct list *oclauses, int var)
{
	char *clause;
	int len;

	len = snprintf(NULL, 0, "%d -%d 0\n", var, var);
	clause = malloc(sizeof(char) * (len+1));
	sprintf(clause, "%d -%d 0\n", var, var);

	return add_list(oclauses, clause);
}

/* qcir must satisfy: "each variable quantified in the prefix must appear in
 * the circuit" - as required by qdimacs standard. preprocess first */
void qcir_to_qdimacs(struct tr *tr, struct dagnode *qcir,
		     FILE *f, char *comment)
{
	struct dagnode *start;
	struct list *clauses, *tmp;
	char *root;
	unsigned int *vars = calloc(2*(tr->hash->next_id+1), sizeof(unsigned int));
	unsigned int *uvars = calloc(2*(tr->hash->next_id+1), sizeof(unsigned int));
	unsigned int i, n, m, n_q=0, len;
	int lastq = -1, qv;
	n = 0;
	m = 0;
	start = qcir;
	while (start->type == TR_EXISTS || start->type == TR_FORALL)
	{
		start = start->child[0];
		n_q++; /* these gate variables aren't in qdimacs */
		       /* thankfully they're highest numbered when we rename */
	}
	clauses = qcir_to_qdimacs_rec(start, NULL, &n, &m, vars, uvars);

	len = snprintf(NULL, 0, "%d 0\n", start->num);
	root = malloc(sizeof(char) * (len+1));
	sprintf(root, "%d 0\n", start->num);
	clauses = add_list(clauses, root);
	m++;

	/* handle any quantified but unused variables, should do this earlier */
	start = qcir;
	while (start->type == TR_EXISTS || start->type == TR_FORALL)
	{
		for (i=0; i<start->n_vars; i++)
		{
			qv = atoi(start->vars[i]);
			if (uvars[qv] == 0) /* not used but quantified */
			{
				clauses = pointless_clauses(clauses, qv);
				m++;
			}
		}
		start = start->child[0];
	}
	for (i=0; i<n; i++)
		if ( uvars[vars[i]]==0 ) /* not used but quantified */
		{ /* add stupid dummy clauses */
			clauses = pointless_clauses(clauses, vars[i]);
			m++;
		}

	/* print to f */
	if (comment!=NULL)
		fprintf(f, "c %s\n", comment);
	fprintf(f, "c qdimacs conversion by de %s\n",DE_VERSION);
	fprintf(f, "p cnf %u %u", tr->hash->next_id-1-n_q, m);

	/* print quantifier prefix and join consecutive identical quantifiers */
	start = qcir;
	while (start->type == TR_EXISTS || start->type == TR_FORALL)
	{
		if (start->type != lastq) /* new kind of quantifier */
			fprintf(f," %c\n%c", (lastq==-1?' ':'0'),
					     (start->type==TR_EXISTS?'e':'a'));
		for (i=0; i<start->n_vars; i++)
			fprintf(f, " %s", start->vars[i]);

		assert(start->neg == NULL || start->neg[0]==0);
		lastq = start->type;
		start = start->child[0];
	}
	/* now we need to print the variables from CNF conversion */
	if (n>0)
	{
		if (lastq != TR_EXISTS)
			fprintf(f, " 0\ne");
		for (i=0; i<n; i++)
			fprintf(f, " %d", vars[i]);
		fprintf(f, " 0\n");
	}
	else if (lastq != -1) /* good luck, but at least we had a quantifier */
		fprintf(f, " 0\n");

	/* print matrix */
	for (tmp=clauses; tmp; tmp=tmp->next)
		fprintf(f, "%s", (char *)tmp->data);

	free_list(clauses);
	free(vars);
	free(uvars);
	return;
}
