/*
Copyright (c) 2022, Charles Jordan <skip@res.otaru-uc.ac.jp>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/
/* deqbf.h
 * Skip Jordan
 *
 * Type definitions for translating to qcir, etc.
 * chj	 3/29/22		created.
 */

#ifndef DETRTYPESH
#define DETRTYPESH

#include <stdio.h> /* for FILE * */

#ifdef TR_DEBUG2
  #define tr_debug1(a) printf(a)
  #define tr_debug2(a,b) printf(a,b)
  #define tr_debug3(a,b,c) printf(a,b,c)
  #define tr_debug4(a,b,c,d) printf(a,b,c,d)
#else
  #define tr_debug1(a)
  #define tr_debug2(a,b)
  #define tr_debug3(a,b,c)
  #define tr_debug4(a,b,c,d)
#endif


#define TR_EXISTS 1
#define TR_FORALL 2
#define TR_AND    3
#define TR_OR     4
#define TR_XOR    5
#define TR_ITE    6 /* not used for now */
#define TR_NOT    7
#define TR_VAR	  8

/* to avoid re-using the same QBF variables, we keep a counter used in
 * cases like \forall x \exists S ...
 * to get different QBF variables for each of the S.
 * 
 */
struct sovc {
	char *varname; /* name of this SO variable */
	unsigned long cur; /* use this value in QBF variables for this var */
	unsigned long next;/* next value to use when needed */
};

/* likewise, we need to make different TC dagnodes if the TC is inside
 * a quantifier, cases like \E E:2 : TC[x,y:E(x,y)](s,t) where the
 * transitive closure obviously depends on E
 * we could track dependencies more carefully to get probably
 * better performance on (badly written?) queries
 * we also stick a few commonly-used unchanging things here to avoid
 * recomputing theme
 */
struct tcvc {
	struct node *tcform; /* pointer to the TC this is */
	int num;	     /*which TC is this, 1...#TC (in case multiple TC)*/
	int tup_arity;	     /* tuple arity, half the arity of the TC */
	int *sorts;	     /* sorts for the arguments, length tup_arity */
	int **values1, **values2; /* fast interpretation pointers */
	struct relrename **renames1, **renames2; /* corresponding renames */
	unsigned long counter; /* we use distinct SO variables for TC,
				* named _T%d where this counter increments it*/
	unsigned long cur;   /* use this value in hash keys for this TC */
};

/* in TC, we rename the predicate symbols, these give the list of renamings */
struct relrename {		/* treat USES of symbol oldname as newname */
	char *oldname;		/* original predicate symbol name */
	char *newname;		/* new predicate symbol name */
	struct relrename *next;	/* next relname */
	/* note that IF the newname is a second-order variable, sovc will be
	 * non-null, and sovc will give the information on how to name the
	 * variable (in case it's shadowing a vocabulary symbol, earlier
	 * binding, etc, giving distinct QBF variables to avoid confusing
	 * solvers.
	 * IF newname is in the vocabulary and not bound, then sovc is NULL
	 */
};

/* a list of TCs that depend on other nodes - used to know when we should
 * increment the cur counter on the TC.
 * ie, in \forall x,y: TC[z1,z2:E(z1,z2)](x,y) we can re-use the TC
 * dagnodes, but in \forall x,y: TC[z1,z2:E(x,z1)&x=y](...) we can't -
 * the TC depends on the values of x,y
 */
struct tcdeps {
	struct node *form;  /* when variables bound here change */
	struct tcvc **tcvc; /* we need to redo these tc */
	int n_tcvc;	    /* how many in tcvc */
};

struct tr {
	struct env *hash;
	long long hits; /* number of hits in the hash table */
	struct list *keys; /* keys in the hash table to free at end */
	struct sovc *sovc; /* array of counters for second order variables */
	int n_sovc;        /* number of second order variables in array */
	struct tcvc *tcvc; /* array of counters for TC subformulas */
	int n_tcvc;	   /* number of elements (TC subformulas) in tcvc */
	struct relrename *renames; /* list of renamed predicates */
	int abort;	   /* if 1, we want to abort, partially implemented */
	int warn;	   /* flag to avoid multiple identical TC warnings */
	struct tcdeps *tcdeps;
	int n_tcdeps;
	struct list *unused; /* list of dagnodes that had all references removed
			      * must refs before freeing, since still in hash
			      */
	int qcirhack;	   /* are we using qcirhack for true/false constants? */
};

struct dagnode {
	int type; /* one of TR_ values above */
	int num;  /* used for getting gate var names: number of this node */
	struct dagnode **child; /* array of children */
	int n_child; /* number of elements in child */
	char **vars; /* array of variables */
	unsigned long n_vars;  /* number of elements in vars */
	int *neg;    /* array of boolean flags for vars/child, 0:pos, 1:negate*/
	int printed; /* Boolean: 1 if already printed, used for QCIR to get
		      * definitions before use, since may not be a tree
		      * value 2 used to make a list of dagnodes for freeing
		      */
	struct dagnode **parents; /* used only in convert.c, not trans.c */
	int n_parents; /* used only in convert.c, not trans.c */
	int refs;    /* reference counter, used when simplifying */
	int simplified; /* 0/1: have we already simplified children? */
	int qsf; /* -1/0/1: do we have a quantified subformula? (convert.c)
		  * 0 unchecked, -1:no, 1:yes
		  */
	int defanged; /* 0/1/2 in convert.c, 1:have we already expanded any
		       * re-used children with quantified subformulas?
		       * 2: have we also renamed variables to be distinct?
		       * 5: have already done CNF conversion of this dag
		       */
};

struct dagnode *tr_get_dagnode_v(struct tr *,int,struct dagnode **,int *,long);
struct dagnode *tr_get_dagnode(struct tr *, int, struct dagnode *, 
			       struct dagnode *,int,int);
struct dagnode *qcir_cleanse(struct tr *, struct dagnode *, unsigned long *);
struct dagnode *qcir_prenex(struct tr *, struct dagnode *, unsigned long *);
void qcir_to_qdimacs(struct tr *, struct dagnode *, FILE *, char *);
void free_dagnode(struct dagnode *);

#endif /* DETRTYPESH */
